package util

import (
	"os"
	"path/filepath"
	"runtime"
)

// AutostartKind 表示当前系统可用的自启机制。
type AutostartKind string

const (
	AutostartNone         AutostartKind = "none"
	AutostartLaunchd      AutostartKind = "launchd"
	DefaultFrpcDirName                  = "frpc"
	defaultConfigFileName               = "frpc.toml"
	defaultPIDFileName                  = "frpc.pid"
	defaultLogFileName                  = "frpc.log"
)

// Environment 收集运行所需的关键路径与系统信息。
type Environment struct {
	OS            string
	Arch          string
	BaseDir       string
	FrpcDir       string
	BinaryName    string
	BinaryPath    string
	ConfigPath    string
	PIDFilePath   string
	LogFilePath   string
	AutostartKind AutostartKind
}

// DetectEnvironment 计算当前环境的通用路径与自启方案。
func DetectEnvironment() Environment {
	baseDir, _ := os.Getwd()
	frpcDir := filepath.Join(baseDir, DefaultFrpcDirName)

	binaryName := "frpc"

	env := Environment{
		OS:          runtime.GOOS,
		Arch:        runtime.GOARCH,
		BaseDir:     baseDir,
		FrpcDir:     frpcDir,
		BinaryName:  binaryName,
		BinaryPath:  filepath.Join(frpcDir, binaryName),
		ConfigPath:  filepath.Join(frpcDir, defaultConfigFileName),
		PIDFilePath: filepath.Join(frpcDir, defaultPIDFileName),
		LogFilePath: filepath.Join(frpcDir, defaultLogFileName),
	}

	if runtime.GOOS == "darwin" {
		env.AutostartKind = AutostartLaunchd
	} else {
		env.AutostartKind = AutostartNone
	}

	return env
}
