package install

import (
	_ "embed"
	"errors"
	"fmt"
	"os"
	"path/filepath"

	"frpc-tool/internal/util"
)

var (
	//go:embed assets/frpc
	frpcBinary []byte
)

// ErrUnsupportedPlatform 在当前构建不支持的平台上触发。
var ErrUnsupportedPlatform = errors.New("frpc install only supported on darwin/arm64")

// EnsureFrpcBinary 会在缺失时写出内置 frpc 二进制。
// 如果文件已存在则返回 (false, nil)。
func EnsureFrpcBinary(env util.Environment) (bool, error) {
	if env.OS != "darwin" || env.Arch != "arm64" {
		return false, ErrUnsupportedPlatform
	}

	if len(frpcBinary) == 0 {
		return false, errors.New("embedded frpc binary is empty")
	}

	if _, err := os.Stat(env.BinaryPath); err == nil {
		return false, nil
	} else if !errors.Is(err, os.ErrNotExist) {
		return false, fmt.Errorf("检查 frpc 目标路径失败: %w", err)
	}

	if err := os.MkdirAll(filepath.Dir(env.BinaryPath), 0o755); err != nil {
		return false, fmt.Errorf("创建 frpc 目标目录失败: %w", err)
	}

	if err := os.WriteFile(env.BinaryPath, frpcBinary, 0o755); err != nil {
		return false, fmt.Errorf("写入 frpc 二进制失败: %w", err)
	}

	return true, nil
}
