package main

import (
	"context"
	"flag"
	"fmt"
	"os"

	"frpc-tool/internal/menu"
	"frpc-tool/internal/util"
)

func main() {
	var (
		cmdInstall        = flag.Bool("install", false, "安装内置 frpc 客户端")
		cmdDeployConfig   = flag.Bool("deploy-config", false, "部署配置文件（使用本地模板或指定来源）")
		cmdStart          = flag.Bool("start", false, "启动 frpc")
		cmdStop           = flag.Bool("stop", false, "停止 frpc")
		cmdRestart        = flag.Bool("restart", false, "重启 frpc")
		cmdStatus         = flag.Bool("status", false, "查看 frpc 运行状态")
		cmdEnableService  = flag.Bool("enable-service", false, "安装并启用自启服务")
		cmdDisableService = flag.Bool("disable-service", false, "卸载自启服务")
		cmdMenu           = flag.Bool("menu", false, "进入交互式终端菜单")
		configSource      = flag.String("config-source", "", "配置来源（本地路径或 HTTP/HTTPS URL）")
	)
	flag.Parse()

	ctx := context.Background()
	env := util.DetectEnvironment()

	if env.OS != "darwin" || env.Arch != "arm64" {
		fmt.Fprintln(os.Stderr, "当前工具仅支持在 macOS arm64 环境运行。")
		os.Exit(1)
	}

	if *cmdMenu {
		menu.Run(ctx, env)
		return
	}

	if flag.NFlag() == 0 {
		menu.Run(ctx, env)
		return
	}

	switch {
	case *cmdInstall:
		menu.HandleInstall(env, os.Stdout, os.Stderr)
	case *cmdDeployConfig:
		menu.HandleDeployConfig(ctx, env, *configSource, os.Stdout, os.Stderr)
	case *cmdStart:
		menu.HandleStart(ctx, env, os.Stdout, os.Stderr)
	case *cmdStop:
		menu.HandleStop(ctx, env, os.Stdout, os.Stderr)
	case *cmdRestart:
		menu.HandleRestart(ctx, env, os.Stdout, os.Stderr)
	case *cmdStatus:
		menu.HandleStatus(ctx, env, os.Stdout, os.Stderr)
	case *cmdEnableService:
		menu.HandleEnableService(ctx, env, os.Stdout, os.Stderr)
	case *cmdDisableService:
		menu.HandleDisableService(ctx, env, os.Stdout, os.Stderr)
	default:
		fmt.Fprintln(os.Stderr, "未指定有效操作。使用 -h 查看命令列表。")
		os.Exit(1)
	}
}
