package utils

import (
	"crypto/rand"
	"encoding/hex"
	"fmt"
	"os"
	"path/filepath"
	"regexp"
	"strings"
)

// SecurityUtils 安全工具包
type SecurityUtils struct{}

// NewSecurityUtils 创建安全工具实例
func NewSecurityUtils() *SecurityUtils {
	return &SecurityUtils{}
}

// ValidatePath 验证路径安全性，防止路径遍历攻击
func (s *SecurityUtils) ValidatePath(path string) error {
	// 清理路径
	cleanPath := filepath.Clean(path)

	// 检查是否包含危险字符
	dangerousPatterns := []string{
		"..",  // 父目录遍历
		"~",   // 用户目录
		"$",   // 环境变量
		"`",   // 命令替换
		"<",   // 重定向
		">",   // 重定向
		"|",   // 管道
		"&",   // 后台执行
		";",   // 命令分隔符
	}

	for _, pattern := range dangerousPatterns {
		if strings.Contains(path, pattern) {
			return fmt.Errorf("路径包含危险字符: %s", pattern)
		}
	}

	// 检查绝对路径是否在允许的目录内
	if filepath.IsAbs(cleanPath) {
		isAllowed := false

		// 获取用户主目录
		homeDir, _ := os.UserHomeDir()

		// 允许的目录列表
		allowedDirs := []string{
			"data",
			"config",
			"logs",
			"tmp",
			filepath.Join(os.TempDir()),
			homeDir,
			filepath.Join(homeDir, ".cos-photo"),
			filepath.Join(homeDir, "Library"),
			filepath.Join(homeDir, ".config"),
		}

		for _, allowedDir := range allowedDirs {
			if strings.HasPrefix(cleanPath, allowedDir) {
				isAllowed = true
				break
			}
		}
		if !isAllowed {
			return fmt.Errorf("不允许访问绝对路径: %s", cleanPath)
		}
	}

	return nil
}

// GenerateSecureTempPath 生成安全的临时文件路径
func (s *SecurityUtils) GenerateSecureTempPath(prefix string) string {
	// 生成随机后缀
	randomBytes := make([]byte, 8)
	rand.Read(randomBytes)
	randomSuffix := hex.EncodeToString(randomBytes)

	tempDir := os.TempDir()
	return filepath.Join(tempDir, fmt.Sprintf("%s-%s", prefix, randomSuffix))
}

// SecureWriteFile 安全写入文件
func (s *SecurityUtils) SecureWriteFile(path string, data []byte, perm os.FileMode) error {
	// 验证路径安全性
	if err := s.ValidatePath(path); err != nil {
		return fmt.Errorf("路径安全验证失败: %v", err)
	}

	// 创建临时文件
	tmpPath := s.GenerateSecureTempPath(filepath.Base(path))
	defer os.Remove(tmpPath) // 清理临时文件

	// 写入临时文件
	if err := os.WriteFile(tmpPath, data, perm); err != nil {
		return fmt.Errorf("写入临时文件失败: %v", err)
	}

	// 原子性重命名
	if err := os.Rename(tmpPath, path); err != nil {
		return fmt.Errorf("重命名文件失败: %v", err)
	}

	return nil
}

// ValidateUserInput 验证用户输入
func (s *SecurityUtils) ValidateUserInput(input string, inputType string) error {
	if strings.TrimSpace(input) == "" {
		return fmt.Errorf("输入不能为空")
	}

	switch inputType {
	case "time":
		// 验证时间输入（分钟、小时、天）
		pattern := `^\d+$`
		if !regexp.MustCompile(pattern).MatchString(input) {
			return fmt.Errorf("时间格式无效，必须是正整数")
		}

		// 验证范围（1分钟到365天）
		if len(input) > 0 {
			// 这里可以根据具体业务调整验证逻辑
		}

	case "path":
		// 验证路径输入
		return s.ValidatePath(input)

	case "command":
		// 验证命令输入
		dangerousCommands := []string{
			"rm", "mv", "cp", "chmod", "chown", "sudo", "su", "kill",
			"reboot", "shutdown", "halt", "poweroff", "dd", "format",
			"fdisk", "mkfs", "mount", "umount", "systemctl", "service",
		}

		lowerInput := strings.ToLower(input)
		for _, cmd := range dangerousCommands {
			if strings.Contains(lowerInput, cmd) {
				return fmt.Errorf("不允许使用危险命令: %s", cmd)
			}
		}

	case "filename":
		// 验证文件名
		pattern := `^[a-zA-Z0-9._-]+$`
		if !regexp.MustCompile(pattern).MatchString(input) {
			return fmt.Errorf("文件名只能包含字母、数字、点、下划线和连字符")
		}

	default:
		// 通用验证
		if len(input) > 1000 {
			return fmt.Errorf("输入长度超过限制")
		}
	}

	return nil
}

// SetSecurePermissions 设置安全权限
func (s *SecurityUtils) SetSecurePermissions(path string, fileType string) error {
	var perm os.FileMode

	switch fileType {
	case "config":
		// 配置文件仅所有者可读写
		perm = 0600
	case "data":
		// 数据文件所有者可读写，其他用户只读
		perm = 0644
	case "executable":
		// 可执行文件所有者可执行，其他用户可读
		perm = 0755
	case "temp":
		// 临时文件所有者可读写
		perm = 0600
	default:
		// 默认权限
		perm = 0644
	}

	return os.Chmod(path, perm)
}

// IsSafeToOperate 检查是否安全执行系统操作
func (s *SecurityUtils) IsSafeToOperate(operation string) bool {
	// 危险操作列表
	dangerousOps := []string{
		"systemctl enable",
		"systemctl disable",
		"systemctl start",
		"systemctl stop",
		"chmod 777",
		"chmod +x",
		"chown root",
		"rm -rf /",
		"dd if=",
		":(){ :|:& };:", // fork bomb
	}

	for _, op := range dangerousOps {
		if strings.Contains(strings.ToLower(operation), strings.ToLower(op)) {
			return false
		}
	}

	return true
}

// SecureExecCommand 安全执行命令
func (s *SecurityUtils) SecureExecCommand(command string, args ...string) error {
	// 验证命令安全性
	if !s.IsSafeToOperate(command) {
		return fmt.Errorf("不允许执行危险命令: %s", command)
	}

	// 验证参数
	for i, arg := range args {
		if err := s.ValidateUserInput(arg, "command"); err != nil {
			return fmt.Errorf("参数 %d 安全验证失败: %v", i, err)
		}
	}

	return nil
}