package tasks

import (
	"fmt"
	"os"
	"path/filepath"
	"time"
)

// SchedulerState 调度器状态管理
type SchedulerState struct {
	configDir string
	stateFile string
}

// NewSchedulerState 创建状态管理器
func NewSchedulerState() *SchedulerState {
	homeDir, _ := os.UserHomeDir()
	configDir := filepath.Join(homeDir, ".cos-photo")

	return &SchedulerState{
		configDir: configDir,
		stateFile: filepath.Join(configDir, "scheduler_state.json"),
	}
}

// State 状态结构
type State struct {
	UserStopped     bool      `json:"user_stopped"`
	Scheduled       bool      `json:"scheduled"`
	LastUpdateTime  time.Time `json:"last_update_time"`
	LastStopTime    time.Time `json:"last_stop_time,omitempty"`
	LastStartTime   time.Time `json:"last_start_time,omitempty"`
}

// SetUserStopped 设置用户停止状态
func (s *SchedulerState) SetUserStopped(stopped bool) error {
	state := &State{
		UserStopped:    stopped,
		LastUpdateTime: time.Now(),
	}

	if stopped {
		state.LastStopTime = time.Now()
	} else {
		state.LastStartTime = time.Now()
	}

	return s.saveState(state)
}

// GetUserStopped 获取用户停止状态
func (s *SchedulerState) GetUserStopped() (bool, error) {
	state, err := s.loadState()
	if err != nil {
		if os.IsNotExist(err) {
			return false, nil // 默认未停止
		}
		return false, err
	}

	return state.UserStopped, nil
}

// SetScheduled 设置定时任务状态
func (s *SchedulerState) SetScheduled(scheduled bool) error {
	state, err := s.loadState()
	if err != nil && !os.IsNotExist(err) {
		return err
	}

	// 如果状态文件不存在，创建默认状态
	if err != nil {
		state = &State{
			UserStopped:    false,
			Scheduled:      scheduled,
			LastUpdateTime: time.Now(),
		}
	} else {
		state.Scheduled = scheduled
		state.LastUpdateTime = time.Now()
	}

	return s.saveState(state)
}

// IsScheduled 获取定时任务状态
func (s *SchedulerState) IsScheduled() bool {
	state, err := s.loadState()
	if err != nil {
		return false // 默认未启用定时任务
	}

	return state.Scheduled
}

// GetState 获取完整状态
func (s *SchedulerState) GetState() (*State, error) {
	return s.loadState()
}

// saveState 保存状态到文件
func (s *SchedulerState) saveState(state *State) error {
	// 确保目录存在
	if err := os.MkdirAll(s.configDir, 0755); err != nil {
		return fmt.Errorf("创建配置目录失败: %v", err)
	}

	// 序列化状态
	data := fmt.Sprintf(`{
  "user_stopped": %t,
  "scheduled": %t,
  "last_update_time": "%s",
  "last_stop_time": "%s",
  "last_start_time": "%s"
}`,
		state.UserStopped,
		state.Scheduled,
		state.LastUpdateTime.Format(time.RFC3339),
		state.LastStopTime.Format(time.RFC3339),
		state.LastStartTime.Format(time.RFC3339),
	)

	// 写入文件
	return os.WriteFile(s.stateFile, []byte(data), 0644)
}

// loadState 从文件加载状态
func (s *SchedulerState) loadState() (*State, error) {
	data, err := os.ReadFile(s.stateFile)
	if err != nil {
		return nil, err
	}

	// 简单解析JSON（这里简化处理，实际应该使用encoding/json）
	state := &State{
		UserStopped:    false,
		Scheduled:      false,
		LastUpdateTime: time.Now(),
	}

	// 检查字段值
	if len(data) > 0 {
		content := string(data)
		state.UserStopped = contains(content, `"user_stopped": true`)
		state.Scheduled = contains(content, `"scheduled": true`)
	}

	return state, nil
}

// contains 简单的字符串包含检查
func contains(s, substr string) bool {
	return len(s) >= len(substr) && findSubstring(s, substr)
}

// findSubstring 查找子字符串
func findSubstring(s, substr string) bool {
	for i := 0; i <= len(s)-len(substr); i++ {
		if s[i:i+len(substr)] == substr {
			return true
		}
	}
	return false
}

// ClearState 清除状态
func (s *SchedulerState) ClearState() error {
	return os.Remove(s.stateFile)
}