package tasks

import (
	"bufio"
	"fmt"
	"net"
	"os"
	"path/filepath"
	"strconv"
	"strings"
	"time"

	"pic-photo/utils"
)

// SecureScheduler 安全的定时任务管理器
type SecureScheduler struct {
	crontabPath string
	security    *utils.SecurityUtils
}

// NewSecureScheduler 创建安全的定时任务管理器
func NewSecureScheduler() *SecureScheduler {
	homeDir, _ := os.UserHomeDir()
	userCrontabPath := filepath.Join(homeDir, ".cos-photo", "crontab")

	return &SecureScheduler{
		crontabPath: userCrontabPath,
		security:    utils.NewSecurityUtils(),
	}
}

// RunSecureSchedulerMenu 运行安全定时任务管理菜单
func (s *SecureScheduler) RunSecureSchedulerMenu() error {
	fmt.Println("⏰ 安全定时任务管理")

	// 确保配置目录存在
	homeDir, _ := os.UserHomeDir()
	configDir := filepath.Join(homeDir, ".cos-photo")
	os.MkdirAll(configDir, 0755)

	reader := bufio.NewReader(os.Stdin)

	for {
		// 清屏
		fmt.Print("\033[2J\033[H")

		// 显示安全定时任务管理子菜单
		fmt.Println("\n" + strings.Repeat("=", 50))
		fmt.Println("              安全定时任务管理")
		fmt.Println(strings.Repeat("=", 50))
		fmt.Println("  1. 查看当前定时任务")
		fmt.Println("  2. 添加自动生成任务")
		fmt.Println("  3. 添加数据备份任务")
		fmt.Println("  4. 添加服务监控任务")
		fmt.Println("  5. 启用/禁用定时任务")
		fmt.Println("  6. 删除定时任务")
		fmt.Println("  7. 测试定时任务")
		fmt.Println("  0. 返回主菜单")
		fmt.Println(strings.Repeat("-", 50))
		fmt.Println("💡 所有任务存储在用户目录下，不影响系统安全")
		fmt.Print("请选择操作 (0-7): ")

		input, err := reader.ReadString('\n')
		if err != nil {
			if err.Error() == "EOF" {
				fmt.Println("\n检测到输入结束，返回主菜单...")
				break
			}
			fmt.Printf("输入错误: %v\n", err)
			continue
		}

		input = strings.TrimSpace(input)

		// 检查是否要退出
		if input == "0" || strings.ToLower(input) == "q" {
			fmt.Println("返回主菜单...")
			break
		}

		// 处理空输入
		if input == "" {
			continue
		}

		// 解析输入
		choice, err := strconv.Atoi(input)
		if err != nil {
			fmt.Println("无效输入，请输入数字")
			continue
		}

		// 处理选择
		switch choice {
		case 1:
			s.ViewScheduledTasks()
		case 2:
			s.AddGenerateTask()
		case 3:
			s.AddBackupTask()
		case 4:
			s.AddMonitorTask()
		case 5:
			s.ToggleScheduledTask()
		case 6:
			s.DeleteScheduledTask()
		case 7:
			s.TestScheduledTask()
		default:
			fmt.Printf("无效选项: %d\n", choice)
		}

		// 每次操作后询问是否继续
		if choice != 0 {
			fmt.Print("\n按回车键继续...")
			reader.ReadString('\n')
		}
	}

	return nil
}

// ViewScheduledTasks 查看当前定时任务
func (s *SecureScheduler) ViewScheduledTasks() {
	fmt.Println("\n📋 当前定时任务列表")
	fmt.Println(strings.Repeat("-", 50))

	// 检查crontab文件
	if _, err := os.Stat(s.crontabPath); os.IsNotExist(err) {
		fmt.Println("暂无定时任务")
		return
	}

	data, err := os.ReadFile(s.crontabPath)
	if err != nil {
		fmt.Printf("读取定时任务文件失败: %v\n", err)
		return
	}

	if len(data) == 0 {
		fmt.Println("暂无定时任务")
	} else {
		fmt.Printf("定时任务配置文件: %s\n", s.crontabPath)
		fmt.Println("任务内容:")
		fmt.Println(strings.Repeat("-", 30))
		fmt.Print(string(data))
		fmt.Println(strings.Repeat("-", 30))
		fmt.Println("💡 这些是用户级定时任务，不会影响系统安全")
	}
}

// AddGenerateTask 添加自动生成任务
func (s *SecureScheduler) AddGenerateTask() {
	fmt.Println("\n➕ 添加自动生成任务")
	fmt.Println(strings.Repeat("-", 50))

	reader := bufio.NewReader(os.Stdin)

	fmt.Print("请输入执行间隔 (小时，1-8760): ")
	intervalInput, _ := reader.ReadString('\n')
	interval := strings.TrimSpace(intervalInput)

	// 验证输入
	if err := s.security.ValidateUserInput(interval, "time"); err != nil {
		fmt.Printf("❌ 输入验证失败: %v\n", err)
		return
	}

	// 验证数值范围
	hours, err := strconv.Atoi(interval)
	if err != nil || hours < 1 || hours > 8760 {
		fmt.Println("❌ 执行间隔必须是1-8760之间的整数")
		return
	}

	// 获取可执行文件路径
	exePath, err := os.Executable()
	if err != nil {
		fmt.Printf("❌ 获取可执行文件路径失败: %v\n", err)
		return
	}

	// 验证可执行路径安全性
	if err := s.security.ValidatePath(exePath); err != nil {
		fmt.Printf("❌ 可执行路径安全验证失败: %v\n", err)
		return
	}

	// 生成安全的crontab条目
	homeDir, _ := os.UserHomeDir()
	logPath := filepath.Join(homeDir, ".cos-photo", "generate.log")
	cronExpr := fmt.Sprintf("0 */%s * * * %s generate >> %s 2>&1", interval, exePath, logPath)

	// 保存到安全位置
	if err := s.appendCronTask(cronExpr); err != nil {
		fmt.Printf("❌ 添加定时任务失败: %v\n", err)
		return
	}

	fmt.Printf("✅ 已添加自动生成任务，每 %s 小时执行一次\n", interval)
	fmt.Printf("📝 Cron表达式: %s\n", cronExpr)
	fmt.Printf("📋 日志路径: %s\n", logPath)
	fmt.Println("💡 这是安全的用户级定时任务")
}

// AddBackupTask 添加数据备份任务
func (s *SecureScheduler) AddBackupTask() {
	fmt.Println("\n💾 添加数据备份任务")
	fmt.Println(strings.Repeat("-", 50))

	reader := bufio.NewReader(os.Stdin)

	fmt.Print("请输入备份间隔 (天，1-365): ")
	intervalInput, _ := reader.ReadString('\n')
	interval := strings.TrimSpace(intervalInput)

	// 验证输入
	if err := s.security.ValidateUserInput(interval, "time"); err != nil {
		fmt.Printf("❌ 输入验证失败: %v\n", err)
		return
	}

	// 验证数值范围
	days, err := strconv.Atoi(interval)
	if err != nil || days < 1 || days > 365 {
		fmt.Println("❌ 备份间隔必须是1-365之间的整数")
		return
	}

	// 获取可执行文件路径
	exePath, err := os.Executable()
	if err != nil {
		fmt.Printf("❌ 获取可执行文件路径失败: %v\n", err)
		return
	}

	// 验证可执行路径安全性
	if err := s.security.ValidatePath(exePath); err != nil {
		fmt.Printf("❌ 可执行路径安全验证失败: %v\n", err)
		return
	}

	// 生成安全的crontab条目
	homeDir, _ := os.UserHomeDir()
	logPath := filepath.Join(homeDir, ".cos-photo", "backup.log")
	cronExpr := fmt.Sprintf("0 2 */%s * * %s backup >> %s 2>&1", interval, exePath, logPath)

	// 保存到安全位置
	if err := s.appendCronTask(cronExpr); err != nil {
		fmt.Printf("❌ 添加定时任务失败: %v\n", err)
		return
	}

	fmt.Printf("✅ 已添加数据备份任务，每 %s 天执行一次\n", interval)
	fmt.Printf("📝 Cron表达式: %s\n", cronExpr)
	fmt.Printf("📋 日志路径: %s\n", logPath)
	fmt.Println("💡 这是安全的用户级定时任务")
}

// AddMonitorTask 添加服务监控任务
func (s *SecureScheduler) AddMonitorTask() {
	fmt.Println("\n🔍 添加服务监控任务")
	fmt.Println(strings.Repeat("-", 50))

	reader := bufio.NewReader(os.Stdin)

	fmt.Print("请输入监控间隔 (分钟，5-1440): ")
	intervalInput, _ := reader.ReadString('\n')
	interval := strings.TrimSpace(intervalInput)

	// 验证输入
	if err := s.security.ValidateUserInput(interval, "time"); err != nil {
		fmt.Printf("❌ 输入验证失败: %v\n", err)
		return
	}

	// 验证数值范围
	minutes, err := strconv.Atoi(interval)
	if err != nil || minutes < 5 || minutes > 1440 {
		fmt.Println("❌ 监控间隔必须是5-1440之间的整数")
		return
	}

	// 获取可执行文件路径
	exePath, err := os.Executable()
	if err != nil {
		fmt.Printf("❌ 获取可执行文件路径失败: %v\n", err)
		return
	}

	// 验证可执行路径安全性
	if err := s.security.ValidatePath(exePath); err != nil {
		fmt.Printf("❌ 可执行路径安全验证失败: %v\n", err)
		return
	}

	// 生成安全的crontab条目
	homeDir, _ := os.UserHomeDir()
	logPath := filepath.Join(homeDir, ".cos-photo", "monitor.log")
	cronExpr := fmt.Sprintf("*/%s * * * * %s monitor >> %s 2>&1", interval, exePath, logPath)

	// 保存到安全位置
	if err := s.appendCronTask(cronExpr); err != nil {
		fmt.Printf("❌ 添加定时任务失败: %v\n", err)
		return
	}

	fmt.Printf("✅ 已添加服务监控任务，每 %s 分钟执行一次\n", interval)
	fmt.Printf("📝 Cron表达式: %s\n", cronExpr)
	fmt.Printf("📋 日志路径: %s\n", logPath)
	fmt.Println("💡 这是安全的用户级定时任务")
}

// ToggleScheduledTask 启用/禁用定时任务
func (s *SecureScheduler) ToggleScheduledTask() {
	fmt.Println("\n🔄 启用/禁用定时任务")
	fmt.Println(strings.Repeat("-", 50))
	fmt.Println("💡 用户级定时任务管理功能")
	fmt.Println("可以通过注释/取消注释任务行来启用/禁用任务")

	// 显示当前任务内容
	s.ViewScheduledTasks()

	fmt.Println("\n请手动编辑以下文件来管理任务:")
	fmt.Printf("📁 配置文件: %s\n", s.crontabPath)
	fmt.Println("💡 以 # 开头的行为注释，不会被执行")
	fmt.Println("💡 使用文本编辑器打开上述文件进行编辑")
}

// DeleteScheduledTask 删除定时任务
func (s *SecureScheduler) DeleteScheduledTask() {
	fmt.Println("\n🗑️ 删除定时任务")
	fmt.Println(strings.Repeat("-", 50))

	if _, err := os.Stat(s.crontabPath); os.IsNotExist(err) {
		fmt.Println("❌ 暂无定时任务文件")
		return
	}

	reader := bufio.NewReader(os.Stdin)
	fmt.Print("⚠️  这将删除所有定时任务，确认继续？(y/N): ")

	input, _ := reader.ReadString('\n')
	if strings.ToLower(strings.TrimSpace(input)) != "y" {
		fmt.Println("❌ 操作已取消")
		return
	}

	if err := os.Remove(s.crontabPath); err != nil {
		fmt.Printf("❌ 删除定时任务文件失败: %v\n", err)
		return
	}

	fmt.Println("✅ 已删除所有定时任务")
	fmt.Println("💡 仅删除了用户级配置，系统安全未受影响")
}

// TestScheduledTask 测试定时任务
func (s *SecureScheduler) TestScheduledTask() {
	fmt.Println("\n🧪 测试定时任务")
	fmt.Println(strings.Repeat("-", 50))

	// 测试生成功能
	fmt.Println("📋 测试自动生成功能...")
	s.testGenerateFunction()

	// 测试备份功能
	fmt.Println("💾 测试数据备份功能...")
	s.testBackupFunction()

	// 测试监控功能
	fmt.Println("🔍 测试服务监控功能...")
	s.testMonitorFunction()

	fmt.Println("🧪 定时任务测试完成")
	fmt.Println("💡 所有测试都在用户环境下安全执行")
}

// testGenerateFunction 测试生成功能
func (s *SecureScheduler) testGenerateFunction() {
	// 获取可执行文件路径
	exePath, err := os.Executable()
	if err != nil {
		fmt.Printf("  ❌ 获取可执行文件路径失败: %v\n", err)
		return
	}

	// 安全执行测试命令
	if err := s.security.SecureExecCommand(exePath, "help"); err != nil {
		fmt.Printf("  ❌ 生成功能测试失败: %v\n", err)
	} else {
		fmt.Println("  ✅ 生成功能测试通过")
	}
}

// testBackupFunction 测试备份功能
func (s *SecureScheduler) testBackupFunction() {
	// 确保备份目录存在
	homeDir, _ := os.UserHomeDir()
	backupDir := filepath.Join(homeDir, ".cos-photo", "backup")
	if err := os.MkdirAll(backupDir, 0755); err != nil {
		fmt.Printf("  ❌ 创建备份目录失败: %v\n", err)
		return
	}

	// 检查源文件是否存在
	if _, err := os.Stat("data/list.json"); err != nil {
		fmt.Println("  ⚠️  源文件不存在，跳过备份测试")
		return
	}

	// 执行安全备份
	timestamp := time.Now().Format("20060102-150405")
	backupFile := filepath.Join(backupDir, fmt.Sprintf("list-%s.json", timestamp))

	// 使用安全的文件操作
	if err := s.secureCopyFile("data/list.json", backupFile); err != nil {
		fmt.Printf("  ❌ 备份失败: %v\n", err)
	} else {
		fmt.Printf("  ✅ 备份成功: %s\n", backupFile)
	}
}

// testMonitorFunction 测试监控功能
func (s *SecureScheduler) testMonitorFunction() {
	// 简单的端口检查（安全方式）
	conn, err := net.DialTimeout("tcp", "localhost:9999", 2*time.Second)
	if err != nil {
		fmt.Println("  ❌ 服务未运行")
	} else {
		conn.Close()
		fmt.Println("  ✅ 服务运行正常")
	}
}

// secureCopyFile 安全复制文件
func (s *SecureScheduler) secureCopyFile(src, dst string) error {
	// 验证源文件路径
	if err := s.security.ValidatePath(src); err != nil {
		return fmt.Errorf("源文件路径不安全: %v", err)
	}

	// 验证目标文件路径
	if err := s.security.ValidatePath(dst); err != nil {
		return fmt.Errorf("目标文件路径不安全: %v", err)
	}

	// 读取源文件
	data, err := os.ReadFile(src)
	if err != nil {
		return fmt.Errorf("读取源文件失败: %v", err)
	}

	// 安全写入目标文件
	return s.security.SecureWriteFile(dst, data, 0644)
}

// appendCronTask 添加cron任务到文件（安全版）
func (s *SecureScheduler) appendCronTask(cronExpr string) error {
	// 验证cron表达式
	if err := s.security.ValidateUserInput(cronExpr, "command"); err != nil {
		return fmt.Errorf("cron表达式验证失败: %v", err)
	}

	// 读取现有内容
	var existingCron []byte
	if data, err := os.ReadFile(s.crontabPath); err == nil {
		existingCron = data
	}

	// 添加新任务
	existingCron = append(existingCron, []byte(cronExpr+"\n")...)

	// 安全写入文件
	return s.security.SecureWriteFile(s.crontabPath, existingCron, 0600)
}

// CheckTaskStatus 检查定时任务状态
func (s *SecureScheduler) CheckTaskStatus() bool {
	// 检查是否存在定时任务文件
	if _, err := os.Stat(s.crontabPath); os.IsNotExist(err) {
		return false
	}

	// 读取文件内容检查是否有活跃任务
	data, err := os.ReadFile(s.crontabPath)
	if err != nil {
		return false
	}

	content := string(data)
	// 检查是否有未注释的cos-photo任务
	lines := strings.Split(content, "\n")
	for _, line := range lines {
		line = strings.TrimSpace(line)
		if line != "" && !strings.HasPrefix(line, "#") && strings.Contains(line, "cos-photo") {
			return true
		}
	}

	return false
}

// EnableSimpleTask 启用简单的定时任务（每天凌晨3点）
func (s *SecureScheduler) EnableSimpleTask() error {
	// 获取当前程序路径
	exePath, err := os.Executable()
	if err != nil {
		return fmt.Errorf("获取程序路径失败: %v", err)
	}

	// 确保配置目录存在
	homeDir, _ := os.UserHomeDir()
	configDir := filepath.Join(homeDir, ".cos-photo")
	os.MkdirAll(configDir, 0755)

	// 创建日志路径
	logPath := filepath.Join(configDir, "scheduled.log")

	// 生成cron表达式：每天凌晨3点执行
	// 格式：分钟 小时 日 月 星期 命令
	// 0 3 * * * 表示每天凌晨3:00执行
	// 先生成数据，然后启动服务并保持后台运行
	cronExpr := fmt.Sprintf("0 3 * * * cd %s && %s generate-data && nohup %s start-server >> %s 2>&1 &", filepath.Dir(exePath), exePath, exePath, logPath)

	// 清除现有的cos-photo任务
	s.clearPhotoTasks()

	// 添加新任务
	if err := s.appendCronTask(cronExpr); err != nil {
		return fmt.Errorf("添加定时任务失败: %v", err)
	}

	return nil
}

// DisableAllTasks 禁用所有定时任务
func (s *SecureScheduler) DisableAllTasks() error {
	if _, err := os.Stat(s.crontabPath); os.IsNotExist(err) {
		return nil // 文件不存在，无需操作
	}

	// 读取现有内容
	data, err := os.ReadFile(s.crontabPath)
	if err != nil {
		return fmt.Errorf("读取定时任务文件失败: %v", err)
	}

	// 注释掉所有cos-photo相关的任务
	lines := strings.Split(string(data), "\n")
	var newLines []string

	for _, line := range lines {
		trimmed := strings.TrimSpace(line)
		if trimmed != "" && strings.Contains(trimmed, "cos-photo") && !strings.HasPrefix(trimmed, "#") {
			// 注释掉cos-photo任务
			newLines = append(newLines, "# "+line)
		} else {
			newLines = append(newLines, line)
		}
	}

	// 写回文件
	newContent := strings.Join(newLines, "\n")
	return s.security.SecureWriteFile(s.crontabPath, []byte(newContent), 0600)
}

// clearPhotoTasks 清除所有cos-photo相关的定时任务
func (s *SecureScheduler) clearPhotoTasks() {
	if _, err := os.Stat(s.crontabPath); os.IsNotExist(err) {
		return
	}

	data, err := os.ReadFile(s.crontabPath)
	if err != nil {
		return
	}

	lines := strings.Split(string(data), "\n")
	var newLines []string

	for _, line := range lines {
		trimmed := strings.TrimSpace(line)
		if trimmed != "" && !strings.Contains(trimmed, "cos-photo") {
			newLines = append(newLines, line)
		}
	}

	if len(newLines) > 0 {
		newContent := strings.Join(newLines, "\n")
		s.security.SecureWriteFile(s.crontabPath, []byte(newContent), 0600)
	}
}