package tasks

import (
	"fmt"
	"net"
	"os"
	"os/exec"
	"path/filepath"
	"strconv"
	"sync"
	"syscall"
	"time"
)

// InternalScheduler 内置定时任务管理器
type InternalScheduler struct {
	running    bool
	stopChan   chan struct{}
	mutex      sync.RWMutex
	ticker     *time.Ticker
	exePath    string
	configDir  string
	logPath    string
	userStopped bool // 用户是否手动停止了服务
}


// NewInternalScheduler 创建内置定时任务管理器
func NewInternalScheduler() *InternalScheduler {
	homeDir, _ := os.UserHomeDir()
	configDir := filepath.Join(homeDir, ".cos-photo")

	return &InternalScheduler{
		stopChan:  make(chan struct{}),
		configDir: configDir,
		logPath:   filepath.Join(configDir, "scheduler.log"),
	}
}

// Start 启动内置定时任务
func (s *InternalScheduler) Start() error {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if s.running {
		return fmt.Errorf("定时任务已经在运行")
	}

	// 获取程序路径
	exePath, err := os.Executable()
	if err != nil {
		return fmt.Errorf("获取程序路径失败: %v", err)
	}
	s.exePath = exePath

	// 确保配置目录存在
	if err := os.MkdirAll(s.configDir, 0755); err != nil {
		return fmt.Errorf("创建配置目录失败: %v", err)
	}

	// 启动独立的定时任务守护进程
	err = s.startDaemonProcess()
	if err != nil {
		return fmt.Errorf("启动定时任务守护进程失败: %v", err)
	}

	s.running = true

	fmt.Println("✅ 内置定时任务已启动")
	fmt.Println("📅 将在每天凌晨3:00自动生成图片索引")
	fmt.Println("🔄 定时任务守护进程已在后台运行")

	return nil
}

// startDaemonProcess 启动独立的定时任务守护进程
func (s *InternalScheduler) startDaemonProcess() error {
	s.logMessage("启动定时任务守护进程...")

	// 使用 nohup 在后台启动定时任务守护进程
	cmd := exec.Command("nohup", s.exePath, "run-scheduler-daemon")
	cmd.Dir = filepath.Dir(s.exePath)

	// 创建PID文件
	pidFile := filepath.Join(s.configDir, "scheduler.pid")

	// 启动守护进程
	err := cmd.Start()
	if err != nil {
		return fmt.Errorf("启动守护进程失败: %v", err)
	}

	// 保存PID
	pid := cmd.Process.Pid
	err = os.WriteFile(pidFile, []byte(strconv.Itoa(pid)), 0644)
	if err != nil {
		s.logMessage(fmt.Sprintf("警告: 无法保存PID文件: %v", err))
	}

	s.logMessage(fmt.Sprintf("定时任务守护进程已启动，PID: %d", pid))

	// 不等待进程完成，让它在后台运行
	go func() {
		cmd.Wait() // 等待进程结束（清理僵尸进程）
	}()

	return nil
}

// Stop 停止内置定时任务
func (s *InternalScheduler) Stop() error {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	if !s.running {
		return fmt.Errorf("定时任务未在运行")
	}

	s.running = false

	// 停止守护进程
	err := s.stopDaemonProcess()
	if err != nil {
		s.logMessage(fmt.Sprintf("停止守护进程失败: %v", err))
		return fmt.Errorf("停止守护进程失败: %v", err)
	}

	fmt.Println("✅ 内置定时任务已停止")
	return nil
}

// stopDaemonProcess 停止守护进程
func (s *InternalScheduler) stopDaemonProcess() error {
	pidFile := filepath.Join(s.configDir, "scheduler.pid")

	// 读取PID
	pidData, err := os.ReadFile(pidFile)
	if err != nil {
		if os.IsNotExist(err) {
			s.logMessage("PID文件不存在，守护进程可能已经停止")
			return nil
		}
		return fmt.Errorf("读取PID文件失败: %v", err)
	}

	pidStr := string(pidData)
	pid, err := strconv.Atoi(pidStr)
	if err != nil {
		return fmt.Errorf("无效的PID: %v", err)
	}

	// 发送SIGTERM信号
	s.logMessage(fmt.Sprintf("发送停止信号到进程 %d", pid))
	process, err := os.FindProcess(pid)
	if err != nil {
		return fmt.Errorf("找不到进程: %v", err)
	}

	err = process.Signal(syscall.SIGTERM)
	if err != nil {
		s.logMessage(fmt.Sprintf("发送信号失败: %v", err))
		// 如果发送信号失败，尝试强制杀死
		err = process.Kill()
		if err != nil {
			return fmt.Errorf("无法停止进程: %v", err)
		}
	}

	// 删除PID文件
	os.Remove(pidFile)

	s.logMessage(fmt.Sprintf("定时任务守护进程 %d 已停止", pid))
	return nil
}

// IsRunning 检查定时任务是否运行
func (s *InternalScheduler) IsRunning() bool {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	return s.running
}

// runScheduler 运行定时任务
func (s *InternalScheduler) runScheduler() {
	// 创建状态管理器
	stateManager := NewSchedulerState()

	// 计算下一个凌晨3点的时间
	nextRun := s.GetNextRunTime()

	// 状态检查定时器
	stateCheckTicker := time.NewTicker(5 * time.Second) // 每5秒检查一次状态
	defer stateCheckTicker.Stop()

	for s.running {
		select {
		case <-s.stopChan:
			return
		case <-stateCheckTicker.C:
			// 定期检查用户停止状态
			if userStopped, err := stateManager.GetUserStopped(); err == nil {
				s.mutex.Lock()
				currentStopped := s.userStopped
				s.userStopped = userStopped
				s.mutex.Unlock()

				// 状态发生变化时记录日志
				if currentStopped != userStopped {
					if userStopped {
						s.logMessage("检测到用户停止了服务")
					} else {
						s.logMessage("检测到用户重新启动了服务")
					}
				}
			}
		default:
			// 检查是否到了定时执行时间
			now := time.Now()
			if now.After(nextRun) {
				// 如果当前时间已经超过了下一个执行时间，计算下一天的
				nextRun = s.GetNextRunTime()
			}

			waitDuration := nextRun.Sub(now)
			if waitDuration > 0 {
				// 等待一小段时间，然后继续循环检查状态
				time.Sleep(1 * time.Second)
				continue
			}

			// 检查是否还在运行
			if !s.running {
				return
			}

			// 执行定时任务
			s.executeScheduledTask()

			// 计算下一次执行时间
			nextRun = s.GetNextRunTime()
		}
	}
}

// GetNextRunTime 计算下一个凌晨3点的时间
func (s *InternalScheduler) GetNextRunTime() time.Time {
	now := time.Now()

	// 计算今天的凌晨3点
	today3AM := time.Date(now.Year(), now.Month(), now.Day(), 3, 0, 0, 0, now.Location())

	// 如果已经过了今天的3点，返回明天的3点
	if now.After(today3AM) {
		return today3AM.Add(24 * time.Hour)
	}

	return today3AM
}

// executeScheduledTask 执行定时任务
func (s *InternalScheduler) executeScheduledTask() {
	fmt.Println("🔄 开始执行定时任务...")

	// 记录日志
	s.logMessage("=== 定时任务开始执行 ===")

	// 模拟执行命令：./cos-photo generate-data
	cmd := "generate-data"
	err := s.executeCommand(cmd)
	if err != nil {
		s.logMessage(fmt.Sprintf("❌ 执行命令失败: %v", err))
		fmt.Printf("❌ 定时任务执行失败: %v\n", err)
		return
	}

	s.logMessage("✅ 数据生成完成")

	// 模拟执行命令：./cos-photo start-server
	cmd = "start-server"
	err = s.executeCommand(cmd)
	if err != nil {
		s.logMessage(fmt.Sprintf("❌ 启动服务器失败: %v", err))
		fmt.Printf("❌ 服务器启动失败: %v\n", err)
		return
	}

	s.logMessage("✅ 服务器启动完成")
	fmt.Println("✅ 定时任务执行完成")
}

// executeCommand 执行命令（启动独立的子进程）
func (s *InternalScheduler) executeCommand(command string) error {
	s.logMessage(fmt.Sprintf("开始执行命令: %s", command))

	switch command {
	case "generate-data":
		// 执行数据生成
		s.logMessage("执行数据生成任务")
		cmd := exec.Command(s.exePath, "generate-data")
		cmd.Dir = filepath.Dir(s.exePath)

		// 重定向输出到日志文件
		logFile, _ := os.OpenFile(s.logPath, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0644)
		defer logFile.Close()
		cmd.Stdout = logFile
		cmd.Stderr = logFile

		err := cmd.Run()
		if err != nil {
			return fmt.Errorf("数据生成失败: %v", err)
		}

	case "start-server":
		// 检查用户是否手动停止了服务
		if s.userStopped {
			s.logMessage("检测到用户手动停止了服务，跳过自动启动")
			return nil
		}

		// 启动服务器前先检查服务是否已经运行
		if s.isServerRunning() {
			s.logMessage("服务器已在运行，跳过启动")
			return nil
		}

		s.logMessage("启动Web服务器...")
		cmd := exec.Command(s.exePath, "start-server")
		cmd.Dir = filepath.Dir(s.exePath)

		// 重定向输出到日志文件
		logFile, _ := os.OpenFile(s.logPath, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0644)
		defer logFile.Close()
		cmd.Stdout = logFile
		cmd.Stderr = logFile

		err := cmd.Run()
		if err != nil {
			return fmt.Errorf("启动服务器失败: %v", err)
		}

	default:
		return fmt.Errorf("未知命令: %s", command)
	}

	return nil
}

// isServerRunning 检查Web服务器是否正在运行
func (s *InternalScheduler) isServerRunning() bool {
	// 检查端口9999是否被占用
	conn, err := net.DialTimeout("tcp", "localhost:9999", 1*time.Second)
	if err != nil {
		return false
	}
	conn.Close()
	return true
}

// logMessage 记录日志
func (s *InternalScheduler) logMessage(message string) {
	logEntry := fmt.Sprintf("[%s] %s\n", time.Now().Format("2006-01-02 15:04:05"), message)

	// 追加到日志文件
	file, err := os.OpenFile(s.logPath, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0644)
	if err != nil {
		fmt.Printf("❌ 无法打开日志文件: %v\n", err)
		return
	}
	defer file.Close()

	file.WriteString(logEntry)
}

// GetStatus 获取定时任务状态信息
func (s *InternalScheduler) GetStatus() (bool, string) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	if !s.running {
		return false, "❌ 定时任务已禁用"
	}

	// 检查守护进程是否真的在运行
	isDaemonRunning := s.checkDaemonRunning()
	if !isDaemonRunning {
		s.running = false
		return false, "❌ 定时任务守护进程已停止"
	}

	nextRun := s.GetNextRunTime()
	return true, fmt.Sprintf("✅ 定时任务已启用 (守护进程运行中)\n下次执行: %s", nextRun.Format("2006-01-02 15:04:05"))
}

// checkDaemonRunning 检查守护进程是否真的在运行
func (s *InternalScheduler) checkDaemonRunning() bool {
	pidFile := filepath.Join(s.configDir, "scheduler.pid")

	// 读取PID
	pidData, err := os.ReadFile(pidFile)
	if err != nil {
		return false
	}

	pidStr := string(pidData)
	pid, err := strconv.Atoi(pidStr)
	if err != nil {
		return false
	}

	// 检查进程是否存在
	process, err := os.FindProcess(pid)
	if err != nil {
		return false
	}

	// 发送信号0检查进程是否存在（不会杀死进程）
	err = process.Signal(syscall.Signal(0))
	if err != nil {
		// 进程不存在
		os.Remove(pidFile) // 清理无效的PID文件
		return false
	}

	return true
}

// SetExecutablePath 设置可执行文件路径
func (s *InternalScheduler) SetExecutablePath(path string) {
	s.exePath = path
}

// SetUserStopped 设置用户手动停止服务的状态
func (s *InternalScheduler) SetUserStopped(stopped bool) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.userStopped = stopped
	if stopped {
		s.logMessage("用户手动停止了服务，定时任务将尊重用户意愿")
	} else {
		s.logMessage("用户重新启用了服务，定时任务恢复正常")
	}
}

// GetUserStopped 获取用户停止状态
func (s *InternalScheduler) GetUserStopped() bool {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	return s.userStopped
}

// RunDaemonLoop 运行守护进程主循环
func (s *InternalScheduler) RunDaemonLoop() {
	s.logMessage("定时任务守护进程开始运行")

	// 创建PID文件
	homeDir, _ := os.UserHomeDir()
	pidFile := filepath.Join(homeDir, ".cos-photo", "scheduler.pid")
	pid := os.Getpid()
	os.WriteFile(pidFile, []byte(strconv.Itoa(pid)), 0644)

	// 设置运行状态
	s.mutex.Lock()
	s.running = true
	s.stopChan = make(chan struct{})
	s.mutex.Unlock()

	// 运行调度器循环
	s.runScheduler()

	// 清理PID文件
	os.Remove(pidFile)

	s.logMessage("定时任务守护进程退出")
}