package tasks

import (
	"fmt"
	"os"
	"path/filepath"
	"strconv"
	"syscall"
	"time"
)

// DaemonManager 守护进程管理器
type DaemonManager struct {
	configDir string
	pidFile   string
	logFile   string
	exePath   string
}

// NewDaemonManager 创建守护进程管理器
func NewDaemonManager() *DaemonManager {
	homeDir, _ := os.UserHomeDir()
	configDir := filepath.Join(homeDir, ".cos-photo")

	return &DaemonManager{
		configDir: configDir,
		pidFile:   filepath.Join(configDir, "scheduler.pid"),
		logFile:   filepath.Join(configDir, "daemon.log"),
	}
}

// StartDaemon 启动守护进程
func (d *DaemonManager) StartDaemon() error {
	// 检查是否已经运行
	if d.IsRunning() {
		return fmt.Errorf("守护进程已经在运行")
	}

	// 获取程序路径
	exePath, err := os.Executable()
	if err != nil {
		return fmt.Errorf("获取程序路径失败: %v", err)
	}
	d.exePath = exePath

	// 确保配置目录存在
	if err := os.MkdirAll(d.configDir, 0755); err != nil {
		return fmt.Errorf("创建配置目录失败: %v", err)
	}

	// 启动守护进程
	pid, err := d.createDaemon()
	if err != nil {
		return fmt.Errorf("创建守护进程失败: %v", err)
	}

	// 保存PID
	if err := os.WriteFile(d.pidFile, []byte(strconv.Itoa(pid)), 0644); err != nil {
		d.logMessage(fmt.Sprintf("警告: 无法保存PID文件: %v", err))
	}

	d.logMessage(fmt.Sprintf("守护进程已启动，PID: %d", pid))
	fmt.Printf("✅ 守护进程已启动，PID: %d\n", pid)

	return nil
}

// StopDaemon 停止守护进程
func (d *DaemonManager) StopDaemon() error {
	if !d.IsRunning() {
		return fmt.Errorf("守护进程未运行")
	}

	// 读取PID
	pidData, err := os.ReadFile(d.pidFile)
	if err != nil {
		return fmt.Errorf("读取PID文件失败: %v", err)
	}

	pid, err := strconv.Atoi(string(pidData))
	if err != nil {
		return fmt.Errorf("无效的PID: %v", err)
	}

	// 发送SIGTERM信号
	process, err := os.FindProcess(pid)
	if err != nil {
		return fmt.Errorf("找不到进程: %v", err)
	}

	err = process.Signal(syscall.SIGTERM)
	if err != nil {
		// 如果优雅停止失败，强制杀死
		d.logMessage(fmt.Sprintf("优雅停止失败，强制杀死进程 %d", pid))
		err = process.Kill()
		if err != nil {
			return fmt.Errorf("无法停止进程: %v", err)
		}
	}

	// 删除PID文件
	os.Remove(d.pidFile)

	d.logMessage(fmt.Sprintf("守护进程 %d 已停止", pid))
	fmt.Println("✅ 守护进程已停止")

	return nil
}

// IsRunning 检查守护进程是否运行
func (d *DaemonManager) IsRunning() bool {
	// 检查PID文件是否存在
	pidData, err := os.ReadFile(d.pidFile)
	if err != nil {
		return false
	}

	// 解析PID
	pid, err := strconv.Atoi(string(pidData))
	if err != nil {
		return false
	}

	// 检查进程是否存在
	process, err := os.FindProcess(pid)
	if err != nil {
		return false
	}

	// 发送信号0检查进程（不会杀死进程）
	err = process.Signal(syscall.Signal(0))
	if err != nil {
		// 进程不存在，清理PID文件
		os.Remove(d.pidFile)
		return false
	}

	return true
}

// GetStatus 获取守护进程状态
func (d *DaemonManager) GetStatus() string {
	if d.IsRunning() {
		return "✅ 守护进程运行中"
	}
	return "❌ 守护进程已停止"
}

// createDaemon 创建守护进程（使用更简单的方法）
func (d *DaemonManager) createDaemon() (int, error) {
	// 使用 os.StartProcess 创建守护进程
	args := []string{d.exePath, "run-scheduler-daemon"}
	procAttr := &os.ProcAttr{
		Files: []*os.File{
			nil, // 标准输入关闭
			nil, // 标准输出重定向
			nil, // 标准错误重定向
		},
		Sys: &syscall.SysProcAttr{
			Setsid: true, // 创建新的会话
		},
	}

	process, err := os.StartProcess(d.exePath, args, procAttr)
	if err != nil {
		return 0, fmt.Errorf("启动守护进程失败: %v", err)
	}

	// 父进程不等待子进程，让其独立运行
	d.logMessage(fmt.Sprintf("守护进程已启动，PID: %d", process.Pid))

	return process.Pid, nil
}

// runSchedulerInDaemon 在守护进程中运行定时任务
func (d *DaemonManager) runSchedulerInDaemon() {
	d.logMessage("守护进程开始运行定时任务")

	scheduler := NewInternalScheduler()
	scheduler.SetExecutablePath(d.exePath)
	scheduler.RunDaemonLoop()
}

// logMessage 记录日志消息
func (d *DaemonManager) logMessage(message string) {
	logEntry := fmt.Sprintf("[%s] %s\n", time.Now().Format("2006-01-02 15:04:05"), message)

	// 写入日志文件
	file, err := os.OpenFile(d.logFile, os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0644)
	if err != nil {
		return
	}
	defer file.Close()

	file.WriteString(logEntry)
}

// RestartDaemon 重启守护进程
func (d *DaemonManager) RestartDaemon() error {
	fmt.Println("🔄 正在重启守护进程...")

	// 先停止
	if d.IsRunning() {
		if err := d.StopDaemon(); err != nil {
			fmt.Printf("停止守护进程失败: %v\n", err)
		}
		// 等待进程完全停止
		time.Sleep(2 * time.Second)
	}

	// 再启动
	if err := d.StartDaemon(); err != nil {
		return fmt.Errorf("启动守护进程失败: %v", err)
	}

	fmt.Println("✅ 守护进程重启完成")
	return nil
}