package tasks

import (
	"bytes"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"runtime"
	"strings"

	"pic-photo/utils"
)

// SecureAutoStartManager 安全的自启管理器
type SecureAutoStartManager struct {
	security *utils.SecurityUtils
}

// NewSecureAutoStartManager 创建安全的自启管理器
func NewSecureAutoStartManager() *SecureAutoStartManager {
	return &SecureAutoStartManager{
		security: utils.NewSecurityUtils(),
	}
}

// RunSecureAutoStartMenu 运行安全自启管理菜单（保留原有兼容性）
func (a *SecureAutoStartManager) RunSecureAutoStartMenu() error {
	fmt.Println("🔧 安全自启管理")
	fmt.Println("⚠️  请使用主菜单中的简化自启管理功能")
	return nil
}

// CheckAutoStartStatus 检查自启状态
func (a *SecureAutoStartManager) CheckAutoStartStatus() bool {
	homeDir, _ := os.UserHomeDir()

	switch runtime.GOOS {
	case "darwin":
		// macOS: 检查LaunchAgents文件是否存在
		launchAgentDir := filepath.Join(homeDir, "Library", "LaunchAgents")
		launchAgentPath := filepath.Join(launchAgentDir, "com.cosphoto.plist")

		if _, err := os.Stat(launchAgentPath); err != nil {
			return false
		}

		// 进一步检查LaunchAgent是否真的在运行
		cmd := exec.Command("launchctl", "list", "com.cosphoto")
		output, err := cmd.CombinedOutput()
		if err != nil {
			return false
		}

		// 检查输出中是否包含进程ID
		return len(output) > 0 && !bytes.Contains(output, []byte("could not be found"))

	case "linux":
		// Linux: 检查systemd user service
		systemdDir := filepath.Join(homeDir, ".config", "systemd", "user")
		servicePath := filepath.Join(systemdDir, "cosphoto.service")

		if _, err := os.Stat(servicePath); err == nil {
			// 检查service是否启用
			cmd := exec.Command("systemctl", "--user", "is-enabled", "cosphoto")
			if err := cmd.Run(); err == nil {
				return true
			}
		}

	case "windows":
		// Windows: 检查启动文件夹
		startupDir := filepath.Join(os.Getenv("APPDATA"), "Microsoft", "Windows", "Start Menu", "Programs", "Startup")
		shortcutPath := filepath.Join(startupDir, "cosphoto.lnk")

		if _, err := os.Stat(shortcutPath); err == nil {
			return true
		}
	}

	return false
}

// EnableAutoStart 启用自启
func (a *SecureAutoStartManager) EnableAutoStart() error {
	// 获取当前程序路径
	exePath, err := os.Executable()
	if err != nil {
		return fmt.Errorf("获取程序路径失败: %v", err)
	}

	// 如果是临时路径或测试路径，尝试使用当前工作目录的 cos-photo
	if strings.Contains(exePath, "go-build") || strings.Contains(exePath, "temp") {
		// 尝试查找当前目录或系统路径中的 cos-photo
		possiblePaths := []string{
			"./cos-photo",
			filepath.Join(filepath.Dir(exePath), "cos-photo"),
			"/usr/local/bin/cos-photo",
			"/opt/homebrew/bin/cos-photo",
		}

		for _, path := range possiblePaths {
			if _, err := os.Stat(path); err == nil {
				realPath, err := filepath.Abs(path)
				if err == nil {
					exePath = realPath
					break
				}
			}
		}
	}

	homeDir, _ := os.UserHomeDir()

	switch runtime.GOOS {
	case "darwin":
		return a.enableMacAutoStart(exePath, homeDir)
	case "linux":
		return a.enableLinuxAutoStart(exePath, homeDir)
	case "windows":
		return a.enableWindowsAutoStart(exePath, homeDir)
	default:
		return fmt.Errorf("不支持的操作系统: %s", runtime.GOOS)
	}
}

// DisableAutoStart 禁用自启
func (a *SecureAutoStartManager) DisableAutoStart() error {
	homeDir, _ := os.UserHomeDir()

	switch runtime.GOOS {
	case "darwin":
		return a.disableMacAutoStart(homeDir)
	case "linux":
		return a.disableLinuxAutoStart(homeDir)
	case "windows":
		return a.disableWindowsAutoStart(homeDir)
	default:
		return fmt.Errorf("不支持的操作系统: %s", runtime.GOOS)
	}
}

// enableMacAutoStart macOS启用自启
func (a *SecureAutoStartManager) enableMacAutoStart(exePath, homeDir string) error {
	launchAgentDir := filepath.Join(homeDir, "Library", "LaunchAgents")
	os.MkdirAll(launchAgentDir, 0755)

	launchAgentPath := filepath.Join(launchAgentDir, "com.cosphoto.plist")

	plistContent := fmt.Sprintf(`<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>Label</key>
    <string>com.cosphoto</string>
    <key>ProgramArguments</key>
    <array>
        <string>%s</string>
        <string>daemon</string>
    </array>
    <key>RunAtLoad</key>
    <true/>
    <key>KeepAlive</key>
    <true/>
    <key>WorkingDirectory</key>
    <string>%s</string>
    <key>StandardOutPath</key>
    <string>%s/.cos-photo/autostart.log</string>
    <key>StandardErrorPath</key>
    <string>%s/.cos-photo/autostart.log</string>
</dict>
</plist>`, exePath, filepath.Dir(exePath), homeDir, homeDir)

	return a.security.SecureWriteFile(launchAgentPath, []byte(plistContent), 0644)
}

// disableMacAutoStart macOS禁用自启
func (a *SecureAutoStartManager) disableMacAutoStart(homeDir string) error {
	launchAgentPath := filepath.Join(homeDir, "Library", "LaunchAgents", "com.cosphoto.plist")

	if _, err := os.Stat(launchAgentPath); err == nil {
		return os.Remove(launchAgentPath)
	}

	return nil
}

// enableLinuxAutoStart Linux启用自启
func (a *SecureAutoStartManager) enableLinuxAutoStart(exePath, homeDir string) error {
	systemdDir := filepath.Join(homeDir, ".config", "systemd", "user")
	os.MkdirAll(systemdDir, 0755)

	servicePath := filepath.Join(systemdDir, "cosphoto.service")

	serviceContent := fmt.Sprintf(`[Unit]
Description=CosPhoto Service
After=graphical-session.target

[Service]
Type=simple
ExecStart=%s daemon
Restart=on-failure
RestartSec=5
WorkingDirectory=%s

[Install]
WantedBy=default.target
`, exePath, filepath.Dir(exePath))

	if err := a.security.SecureWriteFile(servicePath, []byte(serviceContent), 0644); err != nil {
		return err
	}

	// 重新加载systemd并启用服务
	cmd := exec.Command("systemctl", "--user", "daemon-reload")
	if err := cmd.Run(); err != nil {
		return fmt.Errorf("重新加载systemd失败: %v", err)
	}

	cmd = exec.Command("systemctl", "--user", "enable", "cosphoto")
	if err := cmd.Run(); err != nil {
		return fmt.Errorf("启用服务失败: %v", err)
	}

	return nil
}

// disableLinuxAutoStart Linux禁用自启
func (a *SecureAutoStartManager) disableLinuxAutoStart(homeDir string) error {
	servicePath := filepath.Join(homeDir, ".config", "systemd", "user", "cosphoto.service")

	// 停用并删除服务
	cmd := exec.Command("systemctl", "--user", "disable", "cosphoto")
	cmd.Run() // 忽略错误，因为服务可能不存在

	if _, err := os.Stat(servicePath); err == nil {
		return os.Remove(servicePath)
	}

	return nil
}

// enableWindowsAutoStart Windows启用自启
func (a *SecureAutoStartManager) enableWindowsAutoStart(exePath, homeDir string) error {
	startupDir := filepath.Join(os.Getenv("APPDATA"), "Microsoft", "Windows", "Start Menu", "Programs", "Startup")
	shortcutPath := filepath.Join(startupDir, "cosphoto.lnk")

	// 使用PowerShell创建快捷方式
	psScript := fmt.Sprintf(`$WshShell = New-Object -comObject WScript.Shell
$Shortcut = $WshShell.CreateShortcut("%s")
$Shortcut.TargetPath = "%s"
$Shortcut.Arguments = "daemon"
$Shortcut.WorkingDirectory = "%s"
$Shortcut.Save()`, shortcutPath, exePath, filepath.Dir(exePath))

	cmd := exec.Command("powershell", "-Command", psScript)
	return cmd.Run()
}

// disableWindowsAutoStart Windows禁用自启
func (a *SecureAutoStartManager) disableWindowsAutoStart(homeDir string) error {
	startupDir := filepath.Join(os.Getenv("APPDATA"), "Microsoft", "Windows", "Start Menu", "Programs", "Startup")
	shortcutPath := filepath.Join(startupDir, "cosphoto.lnk")

	if _, err := os.Stat(shortcutPath); err == nil {
		return os.Remove(shortcutPath)
	}

	return nil
}