package server

import (
	"context"
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"time"

	"pic-photo/assets"
	"pic-photo/config"
	"pic-photo/cos"
	"pic-photo/security"
)

const (
	dataDir    = "data"
	outputFile = "list.json"
)

// StartWebServer 启动内置Web服务器
func StartWebServer(config *config.Config) {
	// 添加安全中间件
	mux := http.NewServeMux()

	// 路由处理
	mux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		// 只允许GET和HEAD方法
		if r.Method != "GET" && r.Method != "HEAD" {
			http.Error(w, "方法不允许", http.StatusMethodNotAllowed)
			return
		}

		// 验证路径安全性
		allowedPath, ok := security.IsAllowedPath(config.BasePath, r.URL.Path)
		if !ok {
			http.NotFound(w, r)
			return
		}

		// 根据路径类型处理请求
		switch allowedPath {
		case "/":
			if config.BasePath == "/" || r.URL.Path == config.BasePath {
				HandleGallery(w, r, config, config.BasePath)
			} else {
				http.Redirect(w, r, config.BasePath, http.StatusMovedPermanently)
			}
		case "/list.json":
			HandleJSON(w, r, config)
		case "/health":
			HandleHealth(w, r, config)
		default:
			http.NotFound(w, r)
		}
	})

	// 包装中间件链：请求限制 -> 安全头
	rateLimitMiddleware := security.RateLimitMiddleware(100, 1*time.Minute) // 100请求/分钟
	handler := security.SecurityHeaders(rateLimitMiddleware(mux))

	// 启动服务器
	fmt.Printf("🌐 Web服务器启动成功！\n")
	fmt.Printf("📍 访问地址: http://%s%s\n", config.ListenAddr, config.BasePath)
	fmt.Printf("💡 提示: 使用 Ctrl+C 停止服务器\n")
	fmt.Printf("🔒 安全模式已启用：路径验证、安全头、请求限制\n")

	log.Fatal(http.ListenAndServe(config.ListenAddr, handler))
}

// HealthStatus 健康检查响应
type HealthStatus struct {
	Status      string              `json:"status"`
	Timestamp   string              `json:"timestamp"`
	Service     string              `json:"service"`
	Version     string              `json:"version"`
	DataFile    bool                `json:"dataFileExists"`
	COS         *cos.ConnectionStatus `json:"cos,omitempty"`
	WebServer   bool                `json:"webServer"`
}

// HandleHealth 处理健康检查请求
func HandleHealth(w http.ResponseWriter, r *http.Request, config *config.Config) {
	w.Header().Set("Content-Type", "application/json; charset=utf-8")

	health := HealthStatus{
		Status:    "ok",
		Timestamp: time.Now().Format(time.RFC3339),
		Service:   "cos-photo-album",
		Version:   "1.0.0",
		WebServer: config.EnableWebServer,
	}

	// 检查数据文件
	if _, err := os.Stat(filepath.Join(dataDir, outputFile)); err == nil {
		health.DataFile = true
	}

	// 如果启用了COS连接检查，执行COS连接测试
	if config.CheckConnection && !config.TestMode {
		client := cos.CreateClient(config)
		cosStatus := cos.CheckConnection(client, config)
		health.COS = &cosStatus

		// 如果COS连接失败，将状态设置为degraded
		if !cosStatus.Connected {
			health.Status = "degraded"
		}
	}

	// 设置缓存头
	w.Header().Set("Cache-Control", "no-cache, no-store, must-revalidate")
	w.Header().Set("Access-Control-Allow-Origin", "*")

	// 返回JSON响应
	w.WriteHeader(http.StatusOK)
	json.NewEncoder(w).Encode(health)
}

// HandleGallery 处理相册页面请求
func HandleGallery(w http.ResponseWriter, r *http.Request, config *config.Config, basePath string) {
	// 获取security中间件生成的nonce
	nonce := w.Header().Get("X-Content-Security-Nonce")
	if nonce == "" {
		// 如果没有nonce，生成一个默认的
		nonce = "default-nonce"
	}

	// 将nonce插入到HTML中
	htmlContent := strings.ReplaceAll(assets.BuiltinHTML,
		"<script>",
		fmt.Sprintf("<script nonce=\"%s\">", nonce))

	w.Header().Set("Content-Type", "text/html; charset=utf-8")
	w.Header().Del("X-Content-Security-Nonce") // 移除临时头
	w.Write([]byte(htmlContent))
}

// isAllowedOrigin 检查是否为允许的来源
func isAllowedOrigin(origin string) bool {
	if origin == "" {
		return true // 允许同源请求
	}

	// 允许的域名列表（可根据实际需求调整）
	allowedDomains := []string{
		"localhost",
		"127.0.0.1",
		// 添加您的生产域名，如：
		// "yourdomain.com",
		// "www.yourdomain.com",
	}

	for _, domain := range allowedDomains {
		if strings.Contains(origin, domain) {
			return true
		}
	}

	return false
}

// HandleJSON 处理JSON数据请求
func HandleJSON(w http.ResponseWriter, r *http.Request, config *config.Config) {
	jsonPath := filepath.Join(dataDir, outputFile)

	// 检查文件是否存在
	if _, err := os.Stat(jsonPath); os.IsNotExist(err) {
		http.Error(w, `{"error": "图片列表数据不存在，请先运行程序生成"}`, http.StatusNotFound)
		return
	}

	// 读取文件
	data, err := os.ReadFile(jsonPath)
	if err != nil {
		// 记录错误但不暴露敏感路径信息
		log.Printf("读取JSON文件失败: %v", err)
		http.Error(w, `{"error": "数据文件读取失败"}`, http.StatusInternalServerError)
		return
	}

	// 验证JSON格式
	if !json.Valid(data) {
		log.Printf("JSON数据格式无效")
		http.Error(w, `{"error": "数据文件格式错误"}`, http.StatusInternalServerError)
		return
	}

	// 设置安全的CORS头
	origin := r.Header.Get("Origin")
	if origin != "" && isAllowedOrigin(origin) {
		w.Header().Set("Access-Control-Allow-Origin", origin)
		w.Header().Set("Access-Control-Allow-Methods", "GET, HEAD")
		w.Header().Set("Access-Control-Allow-Headers", "Content-Type")
		w.Header().Set("Access-Control-Allow-Credentials", "true")
		w.Header().Set("Access-Control-Max-Age", "86400") // 24小时
	}

	// 设置其他响应头
	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.Header().Set("Cache-Control", "public, max-age=300") // 缓存5分钟

	// 记录可疑访问尝试
	if origin != "" && !isAllowedOrigin(origin) {
		log.Printf("可疑跨域访问尝试 - Origin: %s, IP: %s", origin, r.RemoteAddr)
	}

	// 发送数据
	if _, err := w.Write(data); err != nil {
		log.Printf("发送JSON数据失败: %v", err)
	}
}

// StartWebServerWithStop 启动可以优雅停止的Web服务器
func StartWebServerWithStop(config *config.Config, stopChan <-chan struct{}) {
	// 添加安全中间件
	mux := http.NewServeMux()

	// 路由处理
	mux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		// 只允许GET和HEAD方法
		if r.Method != "GET" && r.Method != "HEAD" {
			http.Error(w, "方法不允许", http.StatusMethodNotAllowed)
			return
		}

		// 验证路径安全性
		allowedPath, ok := security.IsAllowedPath(config.BasePath, r.URL.Path)
		if !ok {
			http.NotFound(w, r)
			return
		}

		// 根据路径类型处理请求
		switch allowedPath {
		case "/":
			if config.BasePath == "/" || r.URL.Path == config.BasePath {
				HandleGallery(w, r, config, config.BasePath)
			} else {
				http.Redirect(w, r, config.BasePath, http.StatusMovedPermanently)
			}
		case config.BasePath + "list.json":
			HandleJSON(w, r, config)
		case config.BasePath + "health":
			HandleHealth(w, r, config)
		default:
			http.NotFound(w, r)
		}
	})

	// 包装中间件链：请求限制 -> 安全头
	rateLimitMiddleware := security.RateLimitMiddleware(100, 1*time.Minute) // 100请求/分钟
	handler := security.SecurityHeaders(rateLimitMiddleware(mux))

	// 创建服务器
	srv := &http.Server{
		Addr:         config.ListenAddr,
		Handler:      handler,
		ReadTimeout:  15 * time.Second,
		WriteTimeout: 15 * time.Second,
		IdleTimeout:  60 * time.Second,
	}

	// 启动服务器的goroutine
	go func() {
		if err := srv.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Printf("Web服务器启动失败: %v", err)
		}
	}()

	// 等待停止信号
	<-stopChan

	// 优雅关闭服务器
	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	if err := srv.Shutdown(ctx); err != nil {
		log.Printf("Web服务器关闭失败: %v", err)
	} else {
		log.Println("Web服务器已优雅关闭")
	}
}