package menu

import (
	"bufio"
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"os"
	"os/exec"
	"strconv"
	"strings"
	"time"

	"pic-photo/config"
	"pic-photo/cos"
	"pic-photo/tasks"
)

// MenuOption 菜单选项
type MenuOption struct {
	ID          int
	Name        string
	Description string
	Handler     func() error
}

// MenuManager 菜单管理器
type MenuManager struct {
	options []MenuOption
	reader  *bufio.Reader
}

// isTerminal 检查是否在交互式终端中运行
func isTerminal() bool {
	stat, _ := os.Stdin.Stat()
	return (stat.Mode() & os.ModeCharDevice) != 0
}

// NewMenuManager 创建新的菜单管理器
func NewMenuManager() *MenuManager {
	return &MenuManager{
		options: make([]MenuOption, 0),
		reader:  bufio.NewReader(os.Stdin),
	}
}

// AddOption 添加菜单选项
func (m *MenuManager) AddOption(id int, name, description string, handler func() error) {
	m.options = append(m.options, MenuOption{
		ID:          id,
		Name:        name,
		Description: description,
		Handler:     handler,
	})
}

// ShowMenu 显示菜单
func (m *MenuManager) ShowMenu() {
	fmt.Println("\n" + strings.Repeat("=", 50))
	fmt.Println("🚀 Cos-Photo 腾讯云 COS 相册工具 v1.0")
	fmt.Println(strings.Repeat("=", 50))

	// 显示系统状态
	m.showSystemStatus()

	fmt.Println(strings.Repeat("=", 50))

	for _, option := range m.options {
		if option.Description != "" {
			fmt.Printf("  %d. %s - %s\n", option.ID, option.Name, option.Description)
		} else {
			fmt.Printf("  %d. %s\n", option.ID, option.Name)
		}
	}

	fmt.Printf("  0. 退出管理\n")
	fmt.Println(strings.Repeat("-", 50))
}

// GetInput 获取用户输入，改进错误处理
func (m *MenuManager) GetInput() (string, error) {
	fmt.Print("请选择操作 (0-9): ")

	input, err := m.reader.ReadString('\n')
	if err != nil {
		// 处理EOF错误
		if err.Error() == "EOF" {
			return "0", nil // EOF时默认选择退出
		}
		return "", err
	}

	return strings.TrimSpace(input), nil
}

// Run 运行菜单循环，修复无限循环问题
func (m *MenuManager) Run() {
	for {
		// 清屏（仅在交互式终端中）
		if isTerminal() {
			fmt.Print("\033[2J\033[H") // ANSI escape codes: 清屏并移动光标到左上角
		}

		m.ShowMenu()

		// 获取用户输入
		input, err := m.GetInput()
		if err != nil {
			// 如果是EOF错误，优雅退出
			if err.Error() == "EOF" {
				fmt.Println("\n检测到输入结束，正在退出程序...")
				break
			}
			fmt.Printf("输入错误: %v\n", err)
			continue
		}

		// 检查是否要退出
		if input == "0" || strings.ToLower(input) == "q" || strings.ToLower(input) == "quit" {
			fmt.Println("正在退出管理...")
			break
		}

		// 处理空输入 - 直接重新显示菜单
		if input == "" {
			continue
		}

		// 解析输入
		choice, err := strconv.Atoi(input)
		if err != nil {
			fmt.Println("无效输入，请输入数字")
			continue
		}

		// 查找并执行对应的处理函数
		found := false
		for _, option := range m.options {
			if option.ID == choice {
				found = true
				fmt.Printf("\n正在执行: %s\n", option.Name)
				if err := option.Handler(); err != nil {
					fmt.Printf("执行失败: %v\n", err)
				} else {
					fmt.Printf("执行成功: %s\n", option.Name)
				}
				break
			}
		}

		if !found {
			fmt.Printf("无效选项: %d\n", choice)
		}

		// 每次操作后询问是否继续（仅在交互式终端中）
		if isTerminal() {
			fmt.Print("\n按回车键继续，或输入 'q' 退出...")
			input, err := m.reader.ReadString('\n')
			if err != nil {
				// EOF错误，退出循环
				if err.Error() == "EOF" {
					fmt.Println("\n检测到输入结束，正在退出管理...")
					break
				}
				fmt.Printf("输入错误: %v\n", err)
				continue
			}

			input = strings.TrimSpace(strings.ToLower(input))
			if input == "q" || input == "quit" {
				fmt.Println("正在退出管理...")
				break
			}
		}
	}

	fmt.Println("管理已退出")
}

// RunWithExitChoice 运行菜单循环，默认只退出管理面板
func (m *MenuManager) RunWithExitChoice() bool {
	for {
		// 清屏（仅在交互式终端中）
		if isTerminal() {
			fmt.Print("\033[2J\033[H") // ANSI escape codes: 清屏并移动光标到左上角
		}

		m.ShowMenu()

		// 获取用户输入
		input, err := m.GetInput()
		if err != nil {
			// 如果是EOF错误，只退出管理面板，保持服务运行
			if err.Error() == "EOF" {
				fmt.Println("\n检测到输入结束，正在退出管理...")
				fmt.Println("💡 后台任务将继续运行")
				return false
			}
			fmt.Printf("输入错误: %v\n", err)
			continue
		}

		// 检查是否要退出管理
		if input == "0" {
			fmt.Println("正在退出管理...")
			fmt.Println("💡 后台任务将继续运行")
			return false
		}

		// 处理空输入 - 直接重新显示菜单
		if input == "" {
			continue
		}

		// 解析输入
		choice, err := strconv.Atoi(input)
		if err != nil {
			fmt.Println("无效输入，请输入数字")
			continue
		}

		// 查找并执行对应的处理函数
		found := false
		for _, option := range m.options {
			if option.ID == choice {
				found = true
				fmt.Printf("\n正在执行: %s\n", option.Name)
				if err := option.Handler(); err != nil {
					fmt.Printf("执行失败: %v\n", err)
				} else {
					fmt.Printf("执行成功: %s\n", option.Name)
				}
				break
			}
		}

		if !found {
			fmt.Printf("无效选项: %d\n", choice)
		}

		// 每次操作后询问是否继续（仅在交互式终端中）
		if isTerminal() {
			fmt.Print("\n按回车键继续...")
			input, err := m.reader.ReadString('\n')
			if err != nil {
				// EOF错误，只退出管理面板，保持服务运行
				if err.Error() == "EOF" {
					fmt.Println("\n检测到输入结束，正在退出管理...")
					fmt.Println("💡 后台任务将继续运行")
					return false
				}
				fmt.Printf("输入错误: %v\n", err)
				continue
			}

			input = strings.TrimSpace(strings.ToLower(input))
			if input == "q" || input == "quit" {
				fmt.Println("正在退出管理...")
				fmt.Println("💡 后台任务将继续运行")
				return false
			}
		}
	}
}

// showSystemStatus 显示系统状态
func (m *MenuManager) showSystemStatus() {
	fmt.Println("📊 系统状态")

	// 检查服务状态
	serviceStatus := m.checkServiceStatus()
	fmt.Printf("├── 服务状态: %s\n", serviceStatus)

	// 检查后台任务状态
	backgroundStatus := m.checkBackgroundStatus()
	fmt.Printf("├── 后台任务: %s\n", backgroundStatus)

	// 检查数据状态
	dataStatus := m.checkDataStatus()
	fmt.Printf("├── 数据状态: %s\n", dataStatus)

	// 检查配置状态
	configStatus := m.checkConfigStatus()
	fmt.Printf("├── 配置状态: %s\n", configStatus)

	// 检查COS状态
	cosStatus := m.checkCosStatus()
	fmt.Printf("└── COS状态: %s\n", cosStatus)
}

// checkServiceStatus 检查服务状态
func (m *MenuManager) checkServiceStatus() string {
	// 检查端口9999是否被占用
	conn, err := net.DialTimeout("tcp", "localhost:9999", 1*time.Second)
	if err != nil {
		return "服务停止 ❌"
	}
	conn.Close()

	// 尝试访问健康检查接口
	resp, err := http.Get("http://localhost:9999/health")
	if err != nil {
		return "端口占用 ⚠️"
	}
	defer resp.Body.Close()

	if resp.StatusCode == 200 {
		return "运行中 ✅"
	}
	return "服务异常 ❌"
}

// checkDataStatus 检查数据状态
func (m *MenuManager) checkDataStatus() string {
	dataPath := "data/list.json"

	// 检查文件是否存在
	info, err := os.Stat(dataPath)
	if os.IsNotExist(err) {
		return "需生成 ❌"
	}

	// 检查文件大小
	if info.Size() == 0 {
		return "数据为空 ⚠️"
	}

	// 检查JSON格式
	data, err := os.ReadFile(dataPath)
	if err != nil {
		return "读取错误 ❌"
	}

	if !json.Valid(data) {
		return "格式错误 ❌"
	}

	// 获取文件大小
	size := float64(info.Size()) / 1024 / 1024 // 转换为MB
	if size < 1 {
		return fmt.Sprintf("就绪 ✅ (文件: %s, 大小: %.1fKB)", dataPath, size*1024)
	}
	return fmt.Sprintf("就绪 ✅ (文件: %s, 大小: %.1fMB)", dataPath, size)
}

// checkConfigStatus 检查配置状态
func (m *MenuManager) checkConfigStatus() string {
	configPath := "config/config.toml"

	// 检查文件是否存在
	info, err := os.Stat(configPath)
	if os.IsNotExist(err) {
		// 配置文件不存在，尝试自动生成
		if err := m.generateDefaultConfig(); err != nil {
			return "生成失败 ❌"
		}
		return "已生成 ✅"
	}

	// 检查文件大小
	if info.Size() == 0 {
		return "配置为空 ⚠️"
	}

	// 获取最后更新时间
	modTime := info.ModTime().Format("2006-01-02 15:04")
	return fmt.Sprintf("正常 ✅ (最后更新: %s)", modTime)
}

// generateDefaultConfig 自动生成默认配置文件
func (m *MenuManager) generateDefaultConfig() error {
	// 需要导入config包来生成配置
	// 这里通过调用main.go中的逻辑来生成配置
	fmt.Println("🔧 检测到配置文件缺失，正在自动生成...")

	// 创建config目录
	if err := os.MkdirAll("config", 0755); err != nil {
		return fmt.Errorf("创建配置目录失败: %v", err)
	}

	// 生成配置内容（简化版注释）
	configContent := `# Cos-Photo 腾讯云 COS 相册工具配置文件

# 腾讯云 COS 配置
secret_id = "AKIDxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"     # 访问密钥 ID
secret_key = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"            # 访问密钥 Key
bucket = "example-bucket-1250000000"                       # 存储桶名称
region = "ap-shanghai"                                      # COS 地域
prefix = "photo/"                                           # 扫描前缀
custom_domain = "https://your-domain.com"                   # 自定义域名（可选）

# Web 服务器配置
enable_web_server = true    # 是否启用 Web 服务器
listen_addr = "localhost:9999"    # 监听地址
base_path = "/"             # URL 基础路径
test_mode = false          # 测试模式：跳过 COS 连接
check_connection = true    # 是否检查 COS 连接
`

	// 写入配置文件
	configPath := "config/config.toml"
	if err := os.WriteFile(configPath, []byte(configContent), 0644); err != nil {
		return fmt.Errorf("写入配置文件失败: %v", err)
	}

	fmt.Printf("✅ 已自动生成配置文件: %s\n", configPath)
	fmt.Println("💡 请编辑配置文件，填入您的腾讯云 COS 凭据后重新运行程序")

	return nil
}

// checkBackgroundStatus 检查后台任务状态
func (m *MenuManager) checkBackgroundStatus() string {
	// 检查自启状态
	autoStartManager := tasks.NewSecureAutoStartManager()
	autoStartEnabled := autoStartManager.CheckAutoStartStatus()

	// 检查daemon进程是否运行
	daemonRunning := m.isDaemonRunning()

	// 检测定时任务是否启用
	schedulerState := tasks.NewSchedulerState()
	scheduled := schedulerState.IsScheduled()

	var status []string

	if autoStartEnabled {
		status = append(status, "开机自启✅")
	}

	// 检查实际的daemon进程状态
	if daemonRunning {
		if scheduled {
			status = append(status, "定时任务运行中✅")
		} else {
			status = append(status, "仅Web服务运行中⚠️")
		}
	} else {
		if scheduled {
			status = append(status, "定时任务已配置❌")
		}
	}

	if len(status) == 0 {
		return "无后台任务 ❌"
	}

	return strings.Join(status, ", ")
}

// isDaemonRunning 检查daemon进程是否运行
func (m *MenuManager) isDaemonRunning() bool {
	// 首先检查端口9999是否开放
	conn, err := net.DialTimeout("tcp", "localhost:9999", 1*time.Second)
	if err == nil {
		conn.Close()
		return true
	}

	// 检查是否有 "cos-photo daemon" 进程在运行
	cmd := exec.Command("pgrep", "-f", "cos-photo daemon")
	if err := cmd.Run(); err == nil {
		return true
	}

	return false
}

// checkCosStatus 检查COS状态
func (m *MenuManager) checkCosStatus() string {
	configPath := "config/config.toml"

	// 检查配置文件是否存在
	if _, err := os.Stat(configPath); os.IsNotExist(err) {
		return "未配置 ⚠️"
	}

	// 尝试加载配置
	cfg, err := config.LoadConfig()
	if err != nil {
		return "配置错误 ❌"
	}

	// 检查是否为测试模式
	if cfg.TestMode {
		return "测试模式 🧪"
	}

	// 执行COS连接测试（使用超时控制避免阻塞菜单显示）
	result := make(chan string, 1)
	go func() {
		if err := cos.TestConnectionMode(cfg); err != nil {
			result <- "连接失败 ❌"
		} else {
			result <- "连接正常 ✅"
		}
	}()

	// 等待最多2秒的连接测试结果
	select {
	case status := <-result:
		return status
	case <-time.After(2 * time.Second):
		return "检测中... ⏳"
	}
}