package main

import (
	"bufio"
	"encoding/json"
	"fmt"
	"log"
	"net"
	"os"
	"os/exec"
	"os/signal"
	"path/filepath"
	"strconv"
	"strings"
	"syscall"
	"time"

	"pic-photo/config"
	"pic-photo/cos"
	"pic-photo/menu"
	"pic-photo/server"
	"pic-photo/tasks"
)

const (
	dataDir    = "data"
	outputFile = "list.json"
	daemonName = "cos-photo-daemon"
)

func main() {
	// 检查命令行参数
	if len(os.Args) > 1 {
		command := os.Args[1]
		switch command {
		case "generate-data":
			// 生成数据模式
			generateDataOnly()
			return
		case "daemon":
			// 守护进程模式 - 这将创建独立的daemon程序
			runDaemonMode()
			return
		default:
			fmt.Printf("未知命令: %s\n", command)
			fmt.Println("可用命令: generate-data, daemon")
			os.Exit(1)
		}
	}

	// 运行交互式菜单模式
	runMenuMode()
}

// runMenuMode 运行交互式菜单模式
func runMenuMode() {
	// 创建菜单管理器
	menuManager := menu.NewMenuManager()

	// 添加菜单选项 - 纯管理功能
	menuManager.AddOption(1, "立即生成", "", generateData)
	menuManager.AddOption(2, "配置管理", "", manageConfig)
	menuManager.AddOption(3, "启动服务", "", startService)
	menuManager.AddOption(4, "停止服务", "", stopService)
	menuManager.AddOption(5, "重启服务", "", restartService)
	menuManager.AddOption(6, "定时管理", "", setupScheduledTasks)
	menuManager.AddOption(7, "自启管理", "", setupAutoStart)
	menuManager.AddOption(8, "服务状态", "", showServiceStatus)

	// 运行菜单循环
	menuManager.RunWithExitChoice()
}

// generateData 生成数据功能
func generateData() error {
	fmt.Println("🔄 立即生成图片列表数据...")
	generateDataOnly()
	return nil
}

// startService 启动服务功能
func startService() error {
	if isDaemonRunning() {
		fmt.Println("⚠️  服务已在运行中")
		return nil
	}

	fmt.Println("🚀 启动服务...")

	// 检查配置文件
	if err := checkConfigExists(); err != nil {
		return err
	}

	// 检查数据文件
	outputPath := filepath.Join(dataDir, outputFile)
	if _, err := os.Stat(outputPath); os.IsNotExist(err) {
		fmt.Printf("⚠️  数据文件 %s 不存在，先生成数据\n", outputPath)
		generateDataOnly()
	}

	// 启动daemon服务
	return startDaemonService()
}

// stopService 停止服务功能
func stopService() error {
	if !isDaemonRunning() {
		fmt.Println("⚠️  服务未运行")
		return nil
	}

	// 检查自启状态
	autoStartManager := tasks.NewSecureAutoStartManager()
	if autoStartManager.CheckAutoStartStatus() {
		fmt.Println("❌ 无法停止服务")
		fmt.Println("🚨 检测到开机自启已启用")
		fmt.Println("💡 解决方案：请先关闭开机自启（菜单 7 → 2）")
		return fmt.Errorf("无法停止服务：请先关闭开机自启")
	}

	fmt.Println("🛑 停止服务...")
	return stopDaemonService()
}

// restartService 重启服务功能
func restartService() error {
	fmt.Println("🔄 重启服务...")

	if err := stopService(); err != nil {
		fmt.Printf("⚠️  停止服务失败: %v\n", err)
	}

	time.Sleep(1 * time.Second)
	return startService()
}

// setupScheduledTasks 定时任务管理功能
func setupScheduledTasks() error {
	fmt.Println("⏰ 定时任务管理")
	fmt.Println(strings.Repeat("=", 40))
	fmt.Println("每天凌晨3点自动生成图片索引")
	fmt.Println()

	// 检查daemon状态
	daemonRunning := isDaemonRunning()
	if daemonRunning {
		fmt.Println("✅ 服务运行中，定时任务可用")
	} else {
		fmt.Println("⚠️  服务未运行，定时任务不可用")
		fmt.Println("💡 请先启动服务（菜单 3）")
		return nil
	}

	// 检查当前定时任务状态
	schedulerState := tasks.NewSchedulerState()
	scheduled := schedulerState.IsScheduled()

	if scheduled {
		fmt.Println("当前状态: ✅ 定时任务已启用")
	} else {
		fmt.Println("当前状态: ❌ 定时任务已禁用")
	}

	fmt.Println()
	fmt.Println("1. 开启定时任务")
	fmt.Println("2. 关闭定时任务")
	fmt.Println("3. 重启定时任务")
	fmt.Println("0. 返回主菜单")
	fmt.Println(strings.Repeat("-", 40))
	fmt.Print("请选择操作 (0-3): ")

	reader := bufio.NewReader(os.Stdin)
	input, err := reader.ReadString('\n')
	if err != nil {
		return err
	}

	choice := strings.TrimSpace(input)
	switch choice {
	case "1":
		if scheduled {
			fmt.Println("⚠️  定时任务已经启用")
		} else {
			if err := schedulerState.SetScheduled(true); err != nil {
				fmt.Printf("❌ 启用定时任务失败: %v\n", err)
			} else {
				fmt.Println("✅ 定时任务已启用")
				fmt.Println("📅 将在每天凌晨3:00自动生成图片索引")
				// 如果daemon正在运行，通知其重新加载配置
				notifyDaemonReload()
			}
		}
	case "2":
		if !scheduled {
			fmt.Println("⚠️  定时任务已经禁用")
		} else {
			if err := schedulerState.SetScheduled(false); err != nil {
				fmt.Printf("❌ 禁用定时任务失败: %v\n", err)
			} else {
				fmt.Println("✅ 定时任务已禁用")
				notifyDaemonReload()
			}
		}
	case "3":
		if err := schedulerState.SetScheduled(false); err != nil {
			fmt.Printf("❌ 禁用定时任务失败: %v\n", err)
		} else {
			time.Sleep(100 * time.Millisecond)
			if err := schedulerState.SetScheduled(true); err != nil {
				fmt.Printf("❌ 启用定时任务失败: %v\n", err)
			} else {
				fmt.Println("✅ 定时任务已重启")
				notifyDaemonReload()
			}
		}
	case "0", "q":
		fmt.Println("返回主菜单...")
	default:
		fmt.Println("❌ 无效选项")
	}

	return nil
}

// setupAutoStart 自启管理功能
func setupAutoStart() error {
	fmt.Println("🚀 自启管理")
	fmt.Println(strings.Repeat("=", 40))
	fmt.Println("开机自动启动cos-photo服务")
	fmt.Println()

	// 检查当前自启状态
	autoStart := tasks.NewSecureAutoStartManager()
	status := autoStart.CheckAutoStartStatus()

	if status {
		fmt.Println("当前状态: ✅ 自启已启用")
		fmt.Println("开机后将自动启动服务")
	} else {
		fmt.Println("当前状态: ❌ 自启已禁用")
	}

	fmt.Println()
	fmt.Println("1. 开启自启")
	fmt.Println("2. 关闭自启")
	fmt.Println("0. 返回主菜单")
	fmt.Println(strings.Repeat("-", 40))
	fmt.Print("请选择操作 (0-2): ")

	reader := bufio.NewReader(os.Stdin)
	input, err := reader.ReadString('\n')
	if err != nil {
		return err
	}

	choice := strings.TrimSpace(input)
	switch choice {
	case "1":
		if status {
			fmt.Println("⚠️  自启已经启用")
		} else {
			if err := autoStart.EnableAutoStart(); err != nil {
				fmt.Printf("❌ 启用自启失败: %v\n", err)
			} else {
				fmt.Println("✅ 自启已启用")
				fmt.Println("🔄 开机后将自动启动服务")
			}
		}
	case "2":
		if !status {
			fmt.Println("⚠️  自启已经禁用")
		} else {
			if err := autoStart.DisableAutoStart(); err != nil {
				fmt.Printf("❌ 禁用自启失败: %v\n", err)
			} else {
				fmt.Println("✅ 自启已禁用")
			}
		}
	case "0", "q":
		fmt.Println("返回主菜单...")
	default:
		fmt.Println("❌ 无效选项")
	}

	return nil
}

// showServiceStatus 显示服务状态
func showServiceStatus() error {
	fmt.Println("📊 服务状态")
	fmt.Println(strings.Repeat("=", 40))

	// 检查daemon状态
	daemonRunning := isDaemonRunning()
	if daemonRunning {
		fmt.Println("🟢 服务状态: 运行中")
		fmt.Println("📍 访问地址: http://localhost:9999")
	} else {
		fmt.Println("🔴 服务状态: 未运行")
	}

	// 检查定时任务状态
	schedulerState := tasks.NewSchedulerState()
	scheduled := schedulerState.IsScheduled()
	if scheduled {
		fmt.Println("⏰ 定时任务: 已启用")
	} else {
		fmt.Println("⏰ 定时任务: 已禁用")
	}

	// 检查自启状态
	autoStart := tasks.NewSecureAutoStartManager()
	autoStartStatus := autoStart.CheckAutoStartStatus()
	if autoStartStatus {
		fmt.Println("🚀 开机自启: 已启用")
	} else {
		fmt.Println("🚀 开机自启: 已禁用")
	}

	fmt.Println(strings.Repeat("-", 40))
	return nil
}

// runDaemonMode 运行守护进程模式（内置daemon功能）
func runDaemonMode() {
	fmt.Println("🚀 Cos-Photo Daemon 启动中...")

	// 检查配置文件
	if err := checkConfigExists(); err != nil {
		fmt.Printf("❌ 配置检查失败: %v\n", err)
		os.Exit(1)
	}

	// 创建必要目录
	homeDir, _ := os.UserHomeDir()
	configDir := filepath.Join(homeDir, ".cos-photo")
	if err := os.MkdirAll(configDir, 0755); err != nil {
		fmt.Printf("❌ 创建配置目录失败: %v\n", err)
		os.Exit(1)
	}

	// 设置日志文件
	logFile, err := os.OpenFile(filepath.Join(configDir, "daemon.log"), os.O_CREATE|os.O_WRONLY|os.O_APPEND, 0644)
	if err != nil {
		fmt.Printf("❌ 创建日志文件失败: %v\n", err)
		os.Exit(1)
	}
	defer logFile.Close()

	log.SetOutput(logFile)
	log.Println("Daemon 启动")

	// 启动Web服务器
	go startWebServer()

	// 启动定时任务调度器
	go startScheduler()

	// 等待信号
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM)

	<-sigChan
	fmt.Println("\n🛑 收到停止信号，正在关闭Daemon...")
	log.Println("Daemon 收到停止信号")

	// 清理资源
	fmt.Println("✅ Daemon 已停止")
	log.Println("Daemon 已停止")
}

// startWebServer 启动Web服务器
func startWebServer() {
	// 加载配置
	cfg, err := config.LoadConfig()
	if err != nil {
		log.Printf("加载配置失败: %v", err)
		return
	}

	// 检查数据文件是否存在，如果不存在则尝试生成
	outputPath := filepath.Join(dataDir, outputFile)
	if _, err := os.Stat(outputPath); os.IsNotExist(err) {
		log.Printf("数据文件不存在: %s，尝试生成", outputPath)
		if err := generateDataIfNeeded(); err != nil {
			log.Printf("生成数据失败: %v", err)
			fmt.Printf("⚠️  数据文件生成失败: %v\n", err)
			return
		}
	}

	// 启动Web服务器
	if cfg.EnableWebServer {
		log.Println("Web服务器启动")
		fmt.Println("✅ Web服务器启动成功")
		fmt.Printf("📍 访问地址: http://%s%s\n", cfg.ListenAddr, cfg.BasePath)
		server.StartWebServer(cfg)
	} else {
		log.Println("Web服务器未启用")
		fmt.Println("❌ 配置文件中未启用Web服务器")
	}
}

// startScheduler 启动定时任务调度器
func startScheduler() {
	// 创建调度器
	scheduler := tasks.NewInternalScheduler()

	// 获取程序路径
	exePath, err := os.Executable()
	if err != nil {
		log.Printf("获取程序路径失败: %v", err)
		return
	}
	scheduler.SetExecutablePath(exePath)

	// 运行调度器主循环
	scheduler.RunDaemonLoop()
}

// generateDataIfNeeded 生成数据（如果需要）
func generateDataIfNeeded() error {
	// 加载配置
	cfg, err := config.LoadConfig()
	if err != nil {
		return fmt.Errorf("加载配置失败: %v", err)
	}

	// 根据测试模式决定是否连接COS
	if cfg.TestMode {
		fmt.Println("🧪 测试模式：跳过COS连接")
		outputPath := filepath.Join(dataDir, outputFile)
		if _, err := os.Stat(outputPath); os.IsNotExist(err) {
			return fmt.Errorf("测试模式：找不到数据文件 %s", outputPath)
		}
		fmt.Printf("✅ 测试模式：使用现有数据文件 %s\n", outputPath)
		return nil
	}

	// 测试COS连接
	if cfg.CheckConnection {
		if err := cos.TestConnectionMode(cfg); err != nil {
			return fmt.Errorf("COS连接检查失败: %v", err)
		}
	}

	// 创建COS客户端
	client := cos.CreateClient(cfg)

	// 遍历COS对象
	fmt.Println("🔗 正在连接腾讯云COS生成图片列表...")
	log.Println("开始生成图片列表")

	fileList, err := cos.ListAllObjects(client, cfg)
	if err != nil {
		return fmt.Errorf("遍历COS对象失败: %v", err)
	}

	// 确保data目录存在
	if err := os.MkdirAll(dataDir, 0755); err != nil {
		return fmt.Errorf("创建数据目录失败: %v", err)
	}

	// 写入JSON文件
	outputPath := filepath.Join(dataDir, outputFile)
	data, err := json.MarshalIndent(fileList, "", "  ")
	if err != nil {
		return fmt.Errorf("序列化文件列表失败: %v", err)
	}

	if err := os.WriteFile(outputPath, data, 0644); err != nil {
		return fmt.Errorf("写入文件失败: %v", err)
	}

	fmt.Printf("✅ 成功生成: 已将 %d 条文件信息写入 %s\n", len(fileList), outputPath)
	log.Printf("成功生成 %d 条文件信息", len(fileList))
	return nil
}

// isDaemonRunning 检查daemon是否运行
func isDaemonRunning() bool {
	// 首先检查端口9999是否开放
	conn, err := net.DialTimeout("tcp", "localhost:9999", 1*time.Second)
	if err == nil {
		conn.Close()
		return true
	}

	// 检查是否有 "cos-photo daemon" 进程在运行
	cmd := exec.Command("pgrep", "-f", "cos-photo daemon")
	if err := cmd.Run(); err == nil {
		return true
	}

	// 也检查包含"cos-photo.*daemon"的进程（更宽松的匹配）
	cmd = exec.Command("pgrep", "-f", "cos-photo")
	if err := cmd.Run(); err == nil {
		// 如果找到cos-photo进程，进一步检查是否是daemon模式
		output, _ := exec.Command("ps", "-p", "$(pgrep -f cos-photo)", "-o", "command").Output()
		if strings.Contains(string(output), "daemon") {
			return true
		}
	}

	return false
}

// findDaemonExecutable 查找daemon可执行文件
func findDaemonExecutable() (string, error) {
	// 获取当前程序路径
	exePath, err := os.Executable()
	if err != nil {
		return "", fmt.Errorf("获取程序路径失败: %v", err)
	}

	// 查找可能的daemon程序
	possiblePaths := []string{
		"./cos-photo-daemon",
		filepath.Join(filepath.Dir(exePath), "cos-photo-daemon"),
		"/usr/local/bin/cos-photo-daemon",
		"/opt/homebrew/bin/cos-photo-daemon",
	}

	// 如果当前程序名为cos-photo，假设daemon在同一目录
	if filepath.Base(exePath) == "cos-photo" {
		possiblePaths = append([]string{filepath.Join(filepath.Dir(exePath), "cos-photo-daemon")}, possiblePaths...)
	}

	for _, path := range possiblePaths {
		if _, err := os.Stat(path); err == nil {
			return path, nil
		}
	}

	// 如果找不到独立的daemon程序，使用当前程序
	return exePath, nil
}

// startDaemonService 启动daemon服务
func startDaemonService() error {
	// 获取当前程序路径
	exePath, err := os.Executable()
	if err != nil {
		return fmt.Errorf("获取程序路径失败: %v", err)
	}

	// 启动内置daemon模式
	cmd := exec.Command(exePath, "daemon")
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	// 设置进程组
	cmd.SysProcAttr = &syscall.SysProcAttr{
		Setpgid: true,
	}

	if err := cmd.Start(); err != nil {
		return fmt.Errorf("启动daemon失败: %v", err)
	}

	// 等待一下让进程启动
	time.Sleep(2 * time.Second)

	// 检查是否启动成功
	if isDaemonRunning() {
		fmt.Println("✅ 服务启动成功")
		fmt.Println("📍 访问地址: http://localhost:9999")
	} else {
		fmt.Println("❌ 服务启动失败")
	}

	return nil
}

// stopDaemonService 停止daemon服务
func stopDaemonService() error {
	// 查找并终止 "cos-photo daemon" 进程
	cmd := exec.Command("pkill", "-f", "cos-photo daemon")
	err := cmd.Run()
	if err != nil {
		// 如果找不到cos-photo daemon，尝试更广的匹配
		cmd = exec.Command("pkill", "-f", "cos-photo")
		cmd.Run()
	}

	// 等待进程停止
	time.Sleep(2 * time.Second)

	if !isDaemonRunning() {
		fmt.Println("✅ 服务已停止")
	} else {
		fmt.Println("❌ 服务停止失败")
	}

	return nil
}

// notifyDaemonReload 通知daemon重新加载配置
func notifyDaemonReload() {
	// 这里可以通过信号或其他方式通知daemon重新加载配置
	// 为简化起见，暂时只记录日志
	fmt.Println("📝 已通知daemon重新加载配置")
}

// checkConfigExists 检查配置文件是否存在
func checkConfigExists() error {
	configPath := "config/config.toml"
	if _, err := os.Stat(configPath); os.IsNotExist(err) {
		fmt.Printf("❌ 配置文件不存在: %s\n", configPath)
		fmt.Println("💡 提示: 请先运行'立即生成'来创建默认配置")
		return fmt.Errorf("配置文件不存在")
	}
	return nil
}

// manageConfig 配置管理功能
func manageConfig() error {
	fmt.Println("⚙️  配置管理")

	// 检查配置文件是否存在
	configPath := "config/config.toml"
	if _, err := os.Stat(configPath); os.IsNotExist(err) {
		fmt.Printf("❌ 配置文件不存在: %s\n", configPath)
		fmt.Println("💡 提示: 请先运行'立即生成'来创建默认配置")
		return nil
	}

	reader := bufio.NewReader(os.Stdin)

	for {
		// 清屏
		fmt.Print("\033[2J\033[H")

		// 显示配置管理子菜单
		fmt.Println("\n" + strings.Repeat("=", 40))
		fmt.Println("           配置管理菜单")
		fmt.Println(strings.Repeat("=", 40))
		fmt.Println("  1. 查看当前配置")
		fmt.Println("  2. 使用默认编辑器编辑配置")
		fmt.Println("  3. 显示配置说明")
		fmt.Println("  4. 测试COS连接")
		fmt.Println("  0. 返回主菜单")
		fmt.Println(strings.Repeat("-", 40))
		fmt.Print("请选择操作 (0-4): ")

		input, err := reader.ReadString('\n')
		if err != nil {
			if err.Error() == "EOF" {
				fmt.Println("\n检测到输入结束，返回主菜单...")
				break
			}
			fmt.Printf("输入错误: %v\n", err)
			continue
		}

		input = strings.TrimSpace(input)

		// 检查是否要退出
		if input == "0" || strings.ToLower(input) == "q" {
			fmt.Println("返回主菜单...")
			break
		}

		// 处理空输入
		if input == "" {
			fmt.Println("无效输入，请输入数字")
			continue
		}

		// 解析输入
		choice, err := strconv.Atoi(input)
		if err != nil {
			fmt.Println("无效输入，请输入数字")
			continue
		}

		// 处理选择
		switch choice {
		case 1:
			showCurrentConfig(configPath)
		case 2:
			editConfigFile(configPath)
		case 3:
			showConfigHelp()
		case 4:
			testCosConnection()
		default:
			fmt.Printf("无效选项: %d\n", choice)
		}

		// 每次操作后询问是否继续
		if choice != 0 {
			fmt.Print("\n按回车键继续...")
			reader.ReadString('\n')
		}
	}

	return nil
}

// showCurrentConfig 显示当前配置
func showCurrentConfig(configPath string) {
	fmt.Printf("📄 配置文件路径: %s\n", configPath)
	fmt.Println(strings.Repeat("-", 50))

	data, err := os.ReadFile(configPath)
	if err != nil {
		fmt.Printf("❌ 读取配置文件失败: %v\n", err)
		return
	}

	fmt.Println(string(data))
	fmt.Println(strings.Repeat("-", 50))
}

// editConfigFile 使用默认编辑器编辑配置文件
func editConfigFile(configPath string) {
	fmt.Printf("📝 正在打开配置文件: %s\n", configPath)

	// 尝试使用系统默认编辑器
	editors := []string{"nano", "vim", "vi", "code", "open"}
	var cmd *exec.Cmd

	for _, editor := range editors {
		if _, err := exec.LookPath(editor); err == nil {
			switch editor {
			case "open":
				// macOS 使用 open 命令
				cmd = exec.Command("open", "-t", configPath)
			case "code":
				// VS Code
				cmd = exec.Command("code", configPath)
			default:
				// 终端编辑器
				cmd = exec.Command(editor, configPath)
			}
			break
		}
	}

	if cmd == nil {
		fmt.Println("❌ 未找到可用的编辑器，请手动编辑配置文件")
		fmt.Printf("配置文件路径: %s\n", configPath)
		return
	}

	// 设置标准输入输出，让编辑器正常工作
	cmd.Stdin = os.Stdin
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	if err := cmd.Run(); err != nil {
		fmt.Printf("❌ 启动编辑器失败: %v\n", err)
		return
	}

	fmt.Println("✅ 配置文件编辑完成")
}

// showConfigHelp 显示配置说明
func showConfigHelp() {
	fmt.Println("📖 配置文件说明")
	fmt.Println(strings.Repeat("=", 50))
	fmt.Println()
	fmt.Println("配置文件格式: TOML")
	fmt.Println("配置文件位置: config/config.toml")
	fmt.Println()
	fmt.Println("🔧 COS 配置项:")
	fmt.Println("  secret_id       - 腾讯云访问密钥 ID")
	fmt.Println("  secret_key      - 腾讯云访问密钥 Key")
	fmt.Println("  bucket          - COS 存储桶名称")
	fmt.Println("  region          - COS 地域 (如: ap-shanghai)")
	fmt.Println("  prefix          - 扫描前缀 (如: photo/)")
	fmt.Println("  custom_domain   - 自定义域名 (可选)")
	fmt.Println()
	fmt.Println("🌐 Web服务器配置项:")
	fmt.Println("  enable_web_server - 是否启用Web服务器 (true/false)")
	fmt.Println("  listen_addr        - 监听地址 (如: localhost:9999)")
	fmt.Println("  base_path          - URL基础路径 (如: /)")
	fmt.Println("  test_mode          - 测试模式，跳过COS连接 (true/false)")
	fmt.Println("  check_connection   - 是否检查COS连接 (true/false)")
	fmt.Println()
	fmt.Println("💡 使用提示:")
	fmt.Println("  1. 首次使用请先配置COS凭据")
	fmt.Println("  2. 测试模式下跳过COS连接，直接使用现有数据")
	fmt.Println("  3. 生产环境建议启用连接检查")
	fmt.Println("  4. 自定义域名可为空，使用COS默认域名")
	fmt.Println(strings.Repeat("=", 50))
}

// testCosConnection 测试COS连接
func testCosConnection() {
	fmt.Println("🔍 测试COS连接...")

	// 加载配置
	cfg, err := config.LoadConfig()
	if err != nil {
		fmt.Printf("❌ 加载配置失败: %v\n", err)
		return
	}

	// 测试连接
	if err := cos.TestConnectionMode(cfg); err != nil {
		fmt.Printf("❌ COS连接测试失败: %v\n", err)
		fmt.Println("💡 请检查配置文件中的COS凭据是否正确")
		return
	}

	fmt.Println("✅ COS连接测试成功")
}

// generateDataOnly 只生成数据，不启动服务器
func generateDataOnly() {
	// 1. 尝试加载配置，如果失败则生成默认配置
	cfg, err := config.LoadConfig()
	if err != nil {
		fmt.Printf("加载配置失败: %v\n", err)
		if err := config.GenerateDefaultConfig(); err != nil {
			fmt.Printf("生成默认配置失败: %v\n", err)
			os.Exit(1)
		}
		fmt.Printf("✅ 已生成默认配置文件: %s\n", filepath.Join("config", "config.toml"))
		fmt.Println("请编辑配置文件后重新运行程序")
		os.Exit(0)
	}

	// 2. 如果启用了连接检查且不是测试模式，先测试COS连接
	if cfg.CheckConnection && !cfg.TestMode {
		fmt.Println("🔍 检查COS连接状态...")
		if err := cos.TestConnectionMode(cfg); err != nil {
			fmt.Printf("❌ COS连接检查失败: %v\n", err)
			fmt.Println("💡 提示: 请检查配置文件中的COS凭据是否正确")
			os.Exit(1)
		}
	}

	// 3. 根据测试模式决定是否连接COS
	if cfg.TestMode {
		fmt.Println("🧪 测试模式：跳过COS连接，使用现有数据文件")

		// 确保data目录存在
		if err := os.MkdirAll(dataDir, 0755); err != nil {
			fmt.Printf("创建数据目录 %s 失败: %v\n", dataDir, err)
			os.Exit(1)
		}

		// 测试模式下必须存在数据文件
		outputPath := filepath.Join(dataDir, outputFile)
		if _, err := os.Stat(outputPath); os.IsNotExist(err) {
			fmt.Printf("❌ 测试模式失败：找不到数据文件 %s\n", outputPath)
			fmt.Printf("请先创建 %s 文件或关闭测试模式\n", outputPath)
			os.Exit(1)
		}

		fmt.Printf("✅ 测试模式：使用现有数据文件 %s\n", outputPath)
		return // 测试模式下直接返回，不需要生成数据
	} else {
		// 创建 COS 客户端
		client := cos.CreateClient(cfg)

		// 遍历COS对象
		fmt.Println("🔗 正在连接腾讯云COS生成图片列表...")
		fileList, err := cos.ListAllObjects(client, cfg)
		if err != nil {
			fmt.Printf("❌ 遍历 COS 对象失败: %v\n", err)
			fmt.Println("💡 提示: 请检查配置文件中的COS凭据是否正确，或启用测试模式")
			os.Exit(1)
		}

		// 确保 data 目录存在
		if err := os.MkdirAll(dataDir, 0755); err != nil {
			fmt.Printf("创建数据目录 %s 失败: %v\n", dataDir, err)
			os.Exit(1)
		}

		// 写入 JSON 文件
		outputPath := filepath.Join(dataDir, outputFile)
		data, err := json.MarshalIndent(fileList, "", "  ")
		if err != nil {
			fmt.Printf("序列化文件列表失败: %v\n", err)
			os.Exit(1)
		}

		if err := os.WriteFile(outputPath, data, 0644); err != nil {
			fmt.Printf("写入文件 %s 失败: %v\n", outputPath, err)
			os.Exit(1)
		}

		fmt.Printf("✅ 成功生成: 已将 %d 条文件信息写入 %s\n", len(fileList), outputPath)
	}
}