package cos

import (
	"context"
	"fmt"
	"mime"
	"net/http"
	"net/url"
	"path/filepath"
	"sort"
	"strings"
	"time"

	"github.com/tencentyun/cos-go-sdk-v5"
	"pic-photo/config"
	"pic-photo/models"
)

// NormalizeDomain 规范化自定义域名
func NormalizeDomain(domain string) string {
	d := strings.TrimSpace(domain)
	if d == "" {
		return ""
	}
	if !strings.HasPrefix(strings.ToLower(d), "http://") && !strings.HasPrefix(strings.ToLower(d), "https://") {
		d = "https://" + d
	}
	// 去掉末尾斜杠
	d = strings.TrimRight(d, "/")
	return d
}

// BuildBaseURL 构建基础 URL
func BuildBaseURL(config *config.Config) string {
	domain := NormalizeDomain(config.CustomDomain)
	if domain != "" {
		return domain + "/"
	}
	return fmt.Sprintf("https://%s.cos.%s.myqcloud.com/", config.Bucket, config.Region)
}

// EncodeKeyToURL 对 COS Key 进行 URL 编码
// 使用 url.QueryEscape 以匹配 JS 的 encodeURIComponent 行为
func EncodeKeyToURL(key string) string {
	parts := strings.Split(key, "/")
	for i, part := range parts {
		parts[i] = url.QueryEscape(part)
	}
	return strings.Join(parts, "/")
}

// GetMimeType 根据文件扩展名获取 MIME 类型
// 改进：添加自定义映射以支持标准库可能遗漏的格式，并处理空扩展名
func GetMimeType(filename string) string {
	// 转换扩展名为小写以确保一致性
	ext := strings.ToLower(filepath.Ext(filename))

	// 自定义扩展名映射 (补充标准库可能的不足)
	// 可以根据实际需要添加更多
	customTypes := map[string]string{
		".heic": "image/heic",
		".heif": "image/heif",
		".webp": "image/webp",
		".avif": "image/avif",
		".jxl":  "image/jxl", // JPEG XL (如果支持)
	}

	// 1. 首先检查自定义映射
	if mimeType, ok := customTypes[ext]; ok {
		return mimeType
	}

	// 2. 如果自定义映射没有，则使用 Go 标准库 mime 包
	mimeType := mime.TypeByExtension(ext)
	if mimeType == "" {
		// 3. 如果标准库也无法识别，返回默认类型
		// 或者可以记录日志，看是哪种文件扩展名未被识别
		// fmt.Printf("警告: 无法识别扩展名 '%s' 的 MIME 类型，文件: %s\n", ext, filename)
		return "application/octet-stream" // 或者返回空字符串 ""
	}

	// 4. 标准库可能返回 "type; charset=utf-8"，我们只需要 type 部分
	if parts := strings.Split(mimeType, ";"); len(parts) > 0 {
		return strings.TrimSpace(parts[0])
	}
	return mimeType
}

// CreateClient 创建 COS 客户端
func CreateClient(config *config.Config) *cos.Client {
	u, _ := url.Parse(fmt.Sprintf("https://%s.cos.%s.myqcloud.com", config.Bucket, config.Region))
	b := &cos.BaseURL{BucketURL: u}
	client := cos.NewClient(b, &http.Client{
		Transport: &cos.AuthorizationTransport{
			SecretID:  config.SecretId,
			SecretKey: config.SecretKey,
		},
	})
	return client
}

// ConnectionStatus COS连接状态
type ConnectionStatus struct {
	Connected    bool   `json:"connected"`
	ErrorMessage string `json:"errorMessage,omitempty"`
	ResponseTime int64  `json:"responseTime,omitempty"`
	Bucket       string `json:"bucket"`
	Region       string `json:"region"`
}

// CheckConnection 检查COS连接状态
func CheckConnection(client *cos.Client, config *config.Config) ConnectionStatus {
	start := time.Now()

	// 尝试获取bucket信息作为连接测试
	_, err := client.Bucket.Head(context.Background(), nil)
	responseTime := time.Since(start).Milliseconds()

	status := ConnectionStatus{
		Connected:    err == nil,
		ResponseTime: responseTime,
		Bucket:       config.Bucket,
		Region:       config.Region,
	}

	if err != nil {
		status.ErrorMessage = fmt.Sprintf("COS连接失败: %v", err)
	}

	return status
}

// TestConnectionMode 测试连接模式 - 仅测试COS连接，不生成数据
func TestConnectionMode(config *config.Config) error {
	// 创建COS客户端
	client := CreateClient(config)

	// 检查连接状态
	status := CheckConnection(client, config)

	if !status.Connected {
		return fmt.Errorf("COS连接测试失败: %s", status.ErrorMessage)
	}

	// 尝试列出少量对象来验证权限
	opt := &cos.BucketGetOptions{
		Prefix:  config.Prefix,
		MaxKeys: 1, // 只获取1个对象进行测试
	}

	_, _, err := client.Bucket.Get(context.Background(), opt)
	if err != nil {
		return fmt.Errorf("COS读取权限验证失败: %v", err)
	}

	return nil
}

// ListAllObjects 递归遍历 COS 对象
func ListAllObjects(client *cos.Client, config *config.Config) ([]models.FileInfo, error) {
	var result []models.FileInfo
	opt := &cos.BucketGetOptions{
		Prefix:  config.Prefix,
		MaxKeys: 1000, // 每次请求最大对象数
	}

	baseURL := BuildBaseURL(config)
	fmt.Printf("使用基础 URL: %s\n", baseURL)
	fmt.Printf("扫描前缀: %s\n", config.Prefix)

	for {
		v, _, err := client.Bucket.Get(context.Background(), opt)
		if err != nil {
			return nil, fmt.Errorf("获取 COS 对象列表失败: %v", err)
		}

		for _, item := range v.Contents {
			key := item.Key
			// 过滤掉目录占位符 (以 / 结尾的)
			if key == "" || strings.HasSuffix(key, "/") {
				continue
			}

			name := filepath.Base(key)
			// 使用改进后的函数获取 MIME 类型
			mimeType := GetMimeType(name)

			// --- 新增调试日志 ---
			// 打印 MIME 类型为空的文件，这很可能是导致前端过滤后数量不同的原因
			if mimeType == "" {
				fmt.Printf("警告: 文件 '%s' (Key: %s) 的 MIME 类型为空\n", name, key)
			}
			// --- 调试日志结束 ---

			encodedKey := EncodeKeyToURL(key)
			fileURL := baseURL + encodedKey

			// item.LastModified 是字符串，直接赋值
			lastModifiedStr := item.LastModified

			info := models.FileInfo{
				URL:          fileURL,
				Name:         name,
				Key:          key,
				Type:         mimeType, // 正确设置 Type 字段
				Size:         item.Size,
				LastModified: lastModifiedStr, // 直接使用字符串
			}
			result = append(result, info)
		}

		// 检查是否还有更多数据需要获取
		if !v.IsTruncated {
			break
		}
		// 设置下次请求的起始位置
		opt.Marker = v.NextMarker
	}

	// 按最后修改时间从新到旧排序
	sort.Slice(result, func(i, j int) bool {
		// 解析时间字符串进行比较
		t1, err1 := time.Parse(time.RFC3339, result[i].LastModified)
		t2, err2 := time.Parse(time.RFC3339, result[j].LastModified)

		// 处理时间解析错误
		if err1 != nil && err2 != nil {
			// 两个都解析失败，按文件名排序
			return result[i].Name < result[j].Name
		} else if err1 != nil {
			// 第一个解析失败，认为第二个较新
			return false
		} else if err2 != nil {
			// 第二个解析失败，认为第一个较新
			return true
		}

		// 正常比较时间
		return t1.After(t2) // 从新到旧
	})

	return result, nil
}