package config

import (
	"fmt"
	"os"
	"path/filepath"
	"strings"

	"github.com/BurntSushi/toml"
)

const (
	configDir  = "config"
	configFile = "config.toml"
)

// Config 定义配置文件结构
type Config struct {
	SecretId     string `json:"SecretId" toml:"secret_id"`
	SecretKey    string `json:"SecretKey" toml:"secret_key"`
	Bucket       string `json:"Bucket" toml:"bucket"`
	Region       string `json:"Region" toml:"region"`
	Prefix       string `json:"Prefix" toml:"prefix"`
	CustomDomain string `json:"CustomDomain" toml:"custom_domain"`

	// Web服务器配置
	EnableWebServer bool   `json:"EnableWebServer" toml:"enable_web_server"` // 是否启用内置Web服务器
	ListenAddr      string `json:"ListenAddr" toml:"listen_addr"`           // 监听地址，如 "localhost:9999"
	BasePath        string `json:"BasePath" toml:"base_path"`               // URL基础路径，如 "/"
	TestMode        bool   `json:"TestMode" toml:"test_mode"`               // 测试模式：跳过COS连接，直接启动Web服务器
	CheckConnection bool   `json:"CheckConnection" toml:"check_connection"` // 是否检查COS连接状态
}

// 默认配置模板（生产环境示例）
var defaultConfig = Config{
	SecretId:     "AKIDxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx", // 腾讯云访问密钥 ID，请替换为您的真实SecretId
	SecretKey:    "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx", // 腾讯云访问密钥 Key，请替换为您的真实SecretKey
	Bucket:       "example-bucket-1250000000",            // COS存储桶名称，格式: bucket-name-appid
	Region:       "ap-shanghai",                         // COS地域，如：ap-beijing, ap-shanghai, ap-guangzhou等
	Prefix:       "photo/",                              // 扫描前缀，如"photo/"表示只扫描photo/目录下的文件
	CustomDomain: "https://your-domain.com",             // 自定义域名（可选），如不需要可留空""

	// Web服务器配置
	EnableWebServer: true,        // 启用内置Web服务器，提供图片展示功能
	ListenAddr:      "localhost:9999",   // 监听地址，localhost:9999表示仅本地访问，0.0.0.0:9999表示所有网络访问
	BasePath:        "/",              // URL基础路径，通常保持"/"
	TestMode:        false,            // 测试模式：true时跳过COS连接，直接使用现有数据文件
	CheckConnection: true,            // 是否检查COS连接状态，生产环境建议启用
}

// LoadConfig 加载配置文件
func LoadConfig() (*Config, error) {
	configPath := filepath.Join(configDir, configFile)
	data, err := os.ReadFile(configPath)
	if err != nil {
		return nil, fmt.Errorf("无法读取配置文件 %s: %v", configPath, err)
	}

	var config Config
	if err := toml.Unmarshal(data, &config); err != nil {
		return nil, fmt.Errorf("无法解析配置文件 %s: %v", configPath, err)
	}

	// 验证配置
	if err := ValidateConfig(&config); err != nil {
		return nil, fmt.Errorf("配置验证失败: %v", err)
	}

	return &config, nil
}

// ValidateConfig 验证配置的安全性和完整性
func ValidateConfig(config *Config) error {
	// 必要字段验证
	if config.SecretId == "" {
		return fmt.Errorf("secret_id 不能为空，请编辑配置文件")
	}
	if config.SecretKey == "" {
		return fmt.Errorf("secret_key 不能为空，请编辑配置文件")
	}
	if config.Bucket == "" {
		return fmt.Errorf("bucket 不能为空，请编辑配置文件")
	}
	if config.Region == "" {
		return fmt.Errorf("region 不能为空，请编辑配置文件")
	}

	// SecretId 格式验证（腾讯云SecretId通常以AKID开头）
	if !strings.HasPrefix(config.SecretId, "AKID") && len(config.SecretId) < 20 {
		return fmt.Errorf("secret_id 格式无效，应该以 'AKID' 开头或长度至少20位")
	}

	// SecretKey 长度验证
	if len(config.SecretKey) < 30 {
		return fmt.Errorf("secret_key 长度不足，至少需要30位字符")
	}

	// Region 格式验证
	if !strings.HasPrefix(config.Region, "ap-") && !strings.HasPrefix(config.Region, "na-") &&
		!strings.HasPrefix(config.Region, "eu-") && !strings.HasPrefix(config.Region, "me-") {
		return fmt.Errorf("region 格式无效，应该以区域前缀开头，如 'ap-shanghai'")
	}

	// Bucket 格式验证
	if !strings.Contains(config.Bucket, "-") {
		return fmt.Errorf("bucket 格式无效，应该包含连字符，如 'example-bucket-1250000000'")
	}

	// ListenAddr 验证
	if config.EnableWebServer && config.ListenAddr == "" {
		return fmt.Errorf("listen_addr 不能为空，当启用Web服务器时")
	}

	// BasePath 验证
	if !strings.HasPrefix(config.BasePath, "/") {
		return fmt.Errorf("base_path 必须以 '/' 开头")
	}

	// 检查是否使用默认示例配置（仅在非测试模式下）
	if !config.TestMode &&
		(strings.Contains(config.SecretId, "xxxxxxxxxxxxxxxx") ||
			strings.Contains(config.SecretKey, "xxxxxxxxxxxxxxxx") ||
			strings.Contains(config.Bucket, "example-bucket")) {
		return fmt.Errorf("检测到示例配置，请使用真实的腾讯云COS凭据")
	}

	return nil
}

// GenerateDefaultConfig 生成默认配置文件模板
// GenerateDefaultConfig 生成带注释的默认配置文件
func GenerateDefaultConfig() error {
	// 确保 config 目录存在
	if err := os.MkdirAll(configDir, 0755); err != nil {
		return fmt.Errorf("创建配置目录 %s 失败: %v", configDir, err)
	}

	configPath := filepath.Join(configDir, configFile)

	// 检查文件是否已存在
	if _, err := os.Stat(configPath); err == nil {
		fmt.Printf("配置文件 %s 已存在，跳过生成。\n", configPath)
		return nil
	} else if !os.IsNotExist(err) {
		return fmt.Errorf("检查配置文件 %s 状态时出错: %v", configPath, err)
	}

	// 生成带注释的TOML配置内容
	configContent := generateCommentedConfig()

	if err := os.WriteFile(configPath, []byte(configContent), 0644); err != nil {
		return fmt.Errorf("写入默认配置文件 %s 失败: %v", configPath, err)
	}

	fmt.Printf("✅ 已生成配置文件: %s\n", configPath)
	fmt.Println("\n📋 配置文件说明:")
	fmt.Println("   1. 请编辑配置文件，填入您的腾讯云 COS 凭据")
	fmt.Println("   2. 以下为配置文件中需要修改的重要项目:")
	fmt.Println("      - secret_id: 替换为您的真实 Secret ID")
	fmt.Println("      - secret_key: 替换为您的真实 Secret Key")
	fmt.Println("      - bucket: 替换为您的真实存储桶名称")
	fmt.Println("      - region: 确认地域是否正确")
	fmt.Println("   3. 编辑完成后重新运行程序即可")

	return nil
}

// generateCommentedConfig 生成简化注释的TOML配置内容
func generateCommentedConfig() string {
	return `# Cos-Photo 腾讯云 COS 相册工具配置文件

# 腾讯云 COS 配置
secret_id = "AKIDxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"     # 访问密钥 ID
secret_key = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"            # 访问密钥 Key
bucket = "example-bucket-1250000000"                       # 存储桶名称
region = "ap-shanghai"                                      # COS 地域
prefix = "photo/"                                           # 扫描前缀
custom_domain = "https://your-domain.com"                   # 自定义域名（可选）

# Web 服务器配置
enable_web_server = true    # 是否启用 Web 服务器
listen_addr = "localhost:9999"    # 监听地址
base_path = "/"             # URL 基础路径
test_mode = false          # 测试模式：跳过 COS 连接
check_connection = true    # 是否检查 COS 连接
`
}