package assets

// BuiltinHTML 内置HTML内容
const BuiltinHTML = `<!DOCTYPE html>
<html lang="zh">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>相册</title>
<style>
  :root {
    --col-width: 250px;
    --gap: 10px;
    --bg: #f6f7f9;
    --card-bg: #eaecef;
  }
  body { margin: 0; background: var(--bg); font-family: system-ui, -apple-system, Segoe UI, Roboto, "PingFang SC", "Noto Sans SC", sans-serif; }
  header {
    position: sticky; top: 0; z-index: 10;
    background: #fff; border-bottom: 1px solid #eee;
    display: flex; align-items: center; justify-content: space-between;
    padding: 8px 12px;
  }
  header .meta { color: #666; font-size: 14px; }
  .gallery {
    column-width: var(--col-width);
    column-gap: var(--gap);
    padding: var(--gap);
  }
  .item {
    break-inside: avoid; margin-bottom: var(--gap);
    background: var(--card-bg); border-radius: 8px; overflow: hidden;
  }
  .thumb {
    width: 100%; display: block; background: #ddd; cursor: zoom-in;
  }
  .skeleton {
    animation: pulse 1.2s ease-in-out infinite;
    background: linear-gradient(90deg, #e9ebee 25%, #f3f4f6 37%, #e9ebee 63%);
    background-size: 400% 100%;
  }
  @keyframes pulse { 0% {background-position: 100% 50%} 100% {background-position: 0 50%} }
  .lightbox {
    position: fixed; inset: 0; background: rgba(0,0,0,0.9);
    display: flex; align-items: center; justify-content: center;
    z-index: 999; visibility: hidden; opacity: 0; transition: opacity .25s;
  }
  .lightbox.active { visibility: visible; opacity: 1; }
  .lightbox img { max-width: 92vw; max-height: 92vh; }
  .load-wrap { display: flex; justify-content: center; padding: 14px; }
  #loadMore {
    background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 10px 16px; cursor: pointer;
  }
  #loadMore[disabled] { opacity: .6; cursor: default; }
</style>
</head>
<body>

<header>
  <div>相册</div>
  <div class="meta"><span id="count">0</span> / <span id="total">0</span></div>
</header>

<div class="gallery" id="gallery"></div>

<div class="load-wrap">
  <button id="loadMore">加载更多</button>
</div>

<div class="lightbox" id="lightbox" aria-hidden="true">
  <img id="lightbox-img" alt="">
</div>

<script>
const FIRST_LOAD = 30;
const BATCH_SIZE = 20;

let items = [];
let loading = false;

const gallery = document.getElementById('gallery');
const btn = document.getElementById('loadMore');
const cnt = document.getElementById('count');
const tot = document.getElementById('total');

// 检测浏览器是否支持 HEIC
function supportsHEIC() {
  const img = new Image();
  img.src = 'data:image/heic;base64,';
  return img.complete && img.naturalWidth !== 0;
}
const heicSupported = supportsHEIC();

// 根据是否支持 HEIC 返回可用的 URL
function getDisplayUrl(url, isThumb = false) {
  const lower = url.toLowerCase();
  if (lower.endsWith('.heic') && !heicSupported) {
    return isThumb
      ? url + '?imageMogr2/format/jpeg/thumbnail/500x'
      : url + '?imageMogr2/format/jpeg';
  } else {
    return isThumb
      ? url + '?imageMogr2/thumbnail/500x'
      : url;
  }
}

// 取列表
fetch('list.json')
  .then(r => r.json())
  .then(data => {
    items = data.filter(x => x.type && x.type.startsWith('image'));
    tot.textContent = items.length;
    loadNext(FIRST_LOAD);
    autoLoadObserver.observe(btn);
  })
  .catch(err => {
    console.error('加载 list.json 失败', err);
    btn.textContent = '加载失败，重试';
  });

// 分批随机渲染
function loadNext(n = BATCH_SIZE) {
  if (loading) return;
  if (items.length === 0) {
    btn.textContent = '已全部加载';
    btn.disabled = true;
    autoLoadObserver.disconnect();
    return;
  }
  loading = true;
  btn.disabled = true;
  btn.textContent = '加载中…';

  // 从剩余 items 中随机抽取 n 个
  const batch = [];
  for (let i = 0; i < n && items.length > 0; i++) {
    const idx = Math.floor(Math.random() * items.length);
    batch.push(items.splice(idx, 1)[0]);
  }

  const frag = document.createDocumentFragment();

  batch.forEach(it => {
    const wrap = document.createElement('div');
    wrap.className = 'item';

    const ph = document.createElement('div');
    ph.style.height = '200px';
    ph.className = 'skeleton';
    wrap.appendChild(ph);

    const img = document.createElement('img');
    img.className = 'thumb';
    img.alt = it.name || '';
    img.loading = 'lazy';
    img.dataset.src = getDisplayUrl(it.url, true);
    img.dataset.full = getDisplayUrl(it.url, false);

    img.addEventListener('click', () => openLightbox(img.dataset.full));
    img.addEventListener('load', () => {
      if (ph && ph.parentNode) ph.parentNode.removeChild(ph);
    });

    wrap.appendChild(img);
    frag.appendChild(wrap);

    imageObserver.observe(img);
  });

  gallery.appendChild(frag);

  cnt.textContent = parseInt(cnt.textContent) + batch.length;

  loading = false;
  btn.disabled = false;
  btn.textContent = items.length === 0 ? '已全部加载' : '加载更多';
  if (items.length === 0) {
    btn.disabled = true;
    autoLoadObserver.disconnect();
  }
}

// 懒加载图片
const imageObserver = new IntersectionObserver((entries, ob) => {
  entries.forEach(e => {
    if (e.isIntersecting) {
      const img = e.target;
      const src = img.dataset.src;
      if (src) {
        img.src = src;
        img.removeAttribute('data-src');
      }
      ob.unobserve(img);
    }
  });
}, { rootMargin: '200px 0px' });

// 自动加载下一批
const autoLoadObserver = new IntersectionObserver((entries) => {
  entries.forEach(e => {
    if (e.isIntersecting && !loading) {
      loadNext();
    }
  });
}, { rootMargin: '400px 0px' });

// 手动加载按钮
btn.addEventListener('click', () => loadNext());

// 预览大图
function openLightbox(url) {
  if (!url) return;
  const lb = document.getElementById('lightbox');
  const im = document.getElementById('lightbox-img');
  im.src = url;
  lb.classList.add('active');
  lb.setAttribute('aria-hidden', 'false');
}
document.getElementById('lightbox').addEventListener('click', () => {
  const lb = document.getElementById('lightbox');
  const im = document.getElementById('lightbox-img');
  im.src = '';
  lb.classList.remove('active');
  lb.setAttribute('aria-hidden', 'true');
});
</script>

</body>
</html>`