# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## 项目概述

这是一个基于腾讯云对象存储 COS 的图片相册生成器。项目包含 Go 后端程序和前端网页展示：

- **后端**：Go 程序连接腾讯云 COS，获取图片列表并生成 JSON 数据文件
- **前端**：响应式 HTML 页面，支持瀑布流布局、懒加载和图片预览

## 架构结构

### 核心组件

1. **main.go** - 主程序文件
   - 配置管理（`config/config.json`）
   - COS SDK 客户端创建
   - 文件列表生成与 JSON 序列化
   - 按修改时间排序（从新到旧）

2. **data/index.html** - 前端相册页面
   - CSS Grid 瀑布流布局
   - 图片懒加载与性能优化
   - HEIC 格式兼容性处理
   - 灯箱预览功能

### 数据流向

1. Go 程序读取 COS 配置
2. 遍历指定前缀下的所有对象
3. 生成包含 URL、类型、大小等信息的 `data/list.json`
4. 前端加载 JSON 文件并渲染图片墙

## 常用命令

### 构建和运行
```bash
# 直接运行程序（生成数据并启动Web服务器）
go run main.go

# 编译为可执行文件
go build -o cos-photo main.go
./cos-photo

# 测试模式：在配置文件中设置 test_mode = true，跳过COS连接，直接启动Web服务器
```

### Web服务器访问
```bash
# 默认访问地址
http://localhost:9999

# JSON数据接口
http://localhost:9999/list.json
```

### 配置管理
```bash
# 程序会在首次运行时自动生成默认配置模板
# 配置文件位置：config/config.toml (TOML格式)
```

### 依赖管理
```bash
# 下载依赖
go mod download

# 整理依赖
go mod tidy
```

## 配置说明

### 配置文件
程序使用 `config/config.toml` 配置文件，采用TOML格式：

**COS配置：**
- `secret_id` / `secret_key`：腾讯云访问凭证
- `bucket`：存储桶名称
- `region`：地域（如 ap-shanghai）
- `prefix`：扫描前缀（如 "photo/"）
- `custom_domain`：可选的自定义域名

**Web服务器配置：**
- `enable_web_server`：是否启用内置Web服务器（默认：false）
- `listen_addr`：监听地址（默认：127.0.0.1:9999）
- `base_path`：URL基础路径（默认：/）
- `test_mode`：测试模式，跳过COS连接（默认：false）

**TOML配置示例：**
```toml
# 腾讯云 COS 配置
secret_id = "AKIDxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
secret_key = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
bucket = "example-bucket-1250000000"
region = "ap-shanghai"
prefix = "photo/"
custom_domain = "https://your-domain.com"

# Web服务器配置
enable_web_server = true
listen_addr = "localhost:9999"
base_path = "/"
test_mode = false
```

## 安全特性

1. **配置文件管理**：通过配置文件管理所有设置，支持安全访问权限控制
2. **路径遍历防护**：严格验证请求路径，防止目录穿越攻击
3. **HTTP安全头**：完整的HTTP安全头配置，包括CSP、HSTS、XSS保护等
4. **输入验证**：严格的配置格式验证，防止恶意输入
5. **安全默认配置**：默认只监听本地接口，关闭不必要功能

## 开发注意事项

1. **MIME 类型处理**：程序包含自定义扩展名映射，支持 HEIC、HEIF、WebP 等现代图片格式
2. **URL 编码**：使用 `url.QueryEscape` 匹配前端的 `encodeURIComponent` 行为
3. **错误处理**：配置验证、COS API 调用都有完整的错误处理，避免信息泄露
4. **前端优化**：支持懒加载、批量加载、随机显示等性能优化策略
5. **Web服务器**：内置HTTP服务器，支持静态文件服务和JSON API，无需nginx
6. **内置HTML**：相册页面HTML已内置到二进制中，无需额外文件
7. **自动生成**：程序自动检测list.json是否存在，不存在则连接COS生成
8. **测试模式**：开发时可在配置文件中设置 `test_mode = true` 跳过COS连接，快速测试前端功能

## 工作流程

1. 程序读取TOML配置文件
2. 检查 `data/list.json` 是否存在
3. **如果不存在**：连接腾讯云COS → 生成图片列表 → 保存到 `data/list.json`
4. **如果已存在**：直接使用现有数据，跳过COS连接
5. 启动内置Web服务器
6. 用户通过浏览器访问相册页面（HTML已内置到二进制中）
7. 前端加载 `list.json` 数据并展示瀑布流相册

## 文件结构

```
pic-photo/
├── main.go              # 主程序
├── go.mod/go.sum        # Go 模块
├── config/
│   └── config.toml      # 配置文件（运行时生成，TOML格式）
├── data/
│   └── list.json        # 生成的图片列表（HTML已内置到二进制中）
└── CLAUDE.md           # 本文件
```