package utils

import (
	"fmt"
	"log"
	"net/url"
	"strings"
)

// IsValidKey 验证文件 key 的安全性
func IsValidKey(key string) bool {
	if key == "" {
		return false
	}

	// 检查路径遍历攻击
	if strings.Contains(key, "..") {
		return false
	}

	// 检查绝对路径
	if strings.HasPrefix(key, "/") {
		return false
	}

	// 检查 Windows 路径分隔符
	if strings.Contains(key, "\\") {
		return false
	}

	// 检查控制字符
	for _, char := range key {
		if char < 32 || char == 127 {
			return false
		}
	}

	return true
}

// GetFileURL 根据文件 Key 和配置生成访问 URL
func GetFileURL(key string, bucket, region, customDomain string) string {
	// 验证 key 不包含恶意字符
	if !IsValidKey(key) {
		log.Printf("警告: 检测到可疑的文件路径: %s", key)
		return "#invalid-path"
	}

	escapedKey := url.PathEscape(key)
	if customDomain != "" {
		return strings.TrimSuffix(customDomain, "/") + "/" + escapedKey
	}
	return fmt.Sprintf("https://%s.cos.%s.myqcloud.com/%s", bucket, region, escapedKey)
}

// MaskSensitive 遮蔽敏感信息
func MaskSensitive(s string) string {
	if s == "" {
		return "<未设置>"
	}
	if len(s) <= 8 {
		return strings.Repeat("*", len(s))
	}
	return s[:4] + strings.Repeat("*", len(s)-8) + s[len(s)-4:]
}
