package service

import (
	"runtime"
	"testing"
)

func TestNewPlatformDetector(t *testing.T) {
	detector := NewPlatformDetector()
	if detector == nil {
		t.Fatal("NewPlatformDetector() returned nil")
	}

	if detector.info == nil {
		t.Fatal("PlatformDetector.info is nil")
	}
}

func TestPlatformDetector_Detect(t *testing.T) {
	detector := NewPlatformDetector()
	info := detector.Detect()

	if info == nil {
		t.Fatal("Detect() returned nil")
	}

	// 测试操作系统类型检测
	switch runtime.GOOS {
	case "darwin":
		if info.OSType != OS_Darwin {
			t.Errorf("Expected OSType to be OS_Darwin, got %v", info.OSType)
		}
		if info.Platform != "macOS" {
			t.Errorf("Expected Platform to be macOS, got %s", info.Platform)
		}
	case "linux":
		if info.OSType != OS_Linux {
			t.Errorf("Expected OSType to be OS_Linux, got %v", info.OSType)
		}
		if info.Platform != "Linux" {
			t.Errorf("Expected Platform to be Linux, got %s", info.Platform)
		}
	default:
		if info.OSType != OS_Unknown {
			t.Errorf("Expected OSType to be OS_Unknown for unsupported OS, got %v", info.OSType)
		}
	}

	// 测试架构检测
	if info.Architecture == "" {
		t.Error("Architecture should not be empty")
	}

	// 测试服务管理器检测
	if info.ServiceMgr == ServiceMgr_Unknown {
		t.Error("ServiceMgr should not be Unknown on supported platforms")
	}
}

func TestPlatformDetector_IsSupported(t *testing.T) {
	detector := NewPlatformDetector()
	detector.Detect()

	supported := detector.IsSupported()

	switch runtime.GOOS {
	case "darwin", "linux":
		if !supported {
			t.Error("Platform should be supported on darwin and linux")
		}
	default:
		if supported {
			t.Error("Platform should not be supported on unsupported OS")
		}
	}
}

func TestPlatformDetector_GetPlatformInfo(t *testing.T) {
	detector := NewPlatformDetector()
	detector.Detect()

	// 测试检测后的情况
	info := detector.GetPlatformInfo()
	if info.OSType == OS_Unknown && (runtime.GOOS == "darwin" || runtime.GOOS == "linux") {
		t.Error("OSType should not be Unknown after detection on supported platforms")
	}

	if info.Platform == "" {
		t.Error("Platform should not be empty after detection")
	}
}

func TestPlatformDetector_GetServiceManagerPath(t *testing.T) {
	detector := NewPlatformDetector()
	detector.Detect()

	path := detector.GetServiceManagerPath()

	if !detector.IsSupported() {
		if path != "" {
			t.Error("ServiceManagerPath should be empty for unsupported platforms")
		}
		return
	}

	if path == "" {
		t.Error("ServiceManagerPath should not be empty for supported platforms")
	}
}

func TestPlatformDetector_GetRecommendedServiceType(t *testing.T) {
	detector := NewPlatformDetector()
	detector.Detect()

	serviceType := detector.GetRecommendedServiceType()

	if !detector.IsSupported() {
		if serviceType != "" {
			t.Error("RecommendedServiceType should be empty for unsupported platforms")
		}
		return
	}

	if serviceType == "" {
		t.Error("RecommendedServiceType should not be empty for supported platforms")
	}
}

func TestPlatformDetector_IsUserLevelRecommended(t *testing.T) {
	detector := NewPlatformDetector()
	detector.Detect()

	userLevel := detector.IsUserLevelRecommended()

	if !detector.IsSupported() {
		if userLevel {
			t.Error("UserLevel should be false for unsupported platforms")
		}
		return
	}

	// 用户级服务应该被推荐（权限最小化原则）
	if !userLevel {
		t.Error("UserLevel should be true for supported platforms (privilege minimization)")
	}
}

// 基准测试
func BenchmarkPlatformDetector_Detect(b *testing.B) {
	detector := NewPlatformDetector()

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		detector.Detect()
	}
}
