package menu

import (
	"context"
	"fmt"
	"time"

	"index-go/config"
	"index-go/core"
	"index-go/utils"
)

// checkCosConnection 检查COS连接状态
func checkCosConnection(cfg *config.Config) string {
	// 快速检查配置是否完整
	if cfg.SecretId == "" || cfg.SecretKey == "" || cfg.Bucket == "" || cfg.Region == "" {
		return "配置不完整"
	}

	// 尝试创建COS客户端并测试连接
	engine, err := core.NewIndexEngine(cfg)
	if err != nil {
		return "连接失败"
	}

	// 设置短超时时间的context
	ctx, cancel := context.WithTimeout(context.Background(), 3*time.Second)
	defer cancel()

	// 尝试列出少量文件来测试连接
	// 这里只测试连接性，不获取大量数据
	_, err = core.ListAllFiles(ctx, engine.GetClient(), "")
	if err != nil {
		return "连接异常"
	}

	// 连接成功
	return "COS连接正常"
}

// ShowBasicStatus 显示基本状态信息（简洁版）
func ShowBasicStatus(cfg *config.Config) {
	// 检查COS连接状态
	cosStatus := checkCosConnection(cfg)

	// 显示简洁的两行状态
	webURL := utils.FormatWebAddress(cfg.WebServer.Address)
	fmt.Printf("系统状态: %s | Web 服务: %s\n", cosStatus, webURL)
	fmt.Printf("索引目录: %s\n", getPrefixStatus(cfg.Prefix))
}

// ShowConfigSummary 显示配置摘要
func ShowConfigSummary(cfg *config.Config) {
	fmt.Println("配置摘要:")
	fmt.Printf("COS凭证: %s (已配置)\n", maskSecret(cfg.SecretId))
	fmt.Printf("存储桶: %s (%s)\n", cfg.Bucket, cfg.Region)
	prefix := cfg.Prefix
	if prefix == "" {
		prefix = "未设置 (扫描根目录)"
	}
	fmt.Printf("索引目录: %s\n", prefix)
	fmt.Printf("自定义域名: %s\n", getDomainStatus(cfg.CustomDomain))
	fmt.Printf("Web 服务: %s\n", utils.FormatWebAddress(cfg.WebServer.Address))
}

// maskSecret 掩码显示敏感信息
func maskSecret(secret string) string {
	if secret == "" {
		return "未配置"
	}
	if len(secret) <= 8 {
		return "***"
	}
	return secret[:4] + "***" + secret[len(secret)-4:]
}

// getDomainStatus 获取域名状态
func getDomainStatus(domain string) string {
	if domain == "" {
		return "未配置 (使用默认COS域名)"
	}
	return domain
}
