package menu

import (
	"bufio"
	"fmt"
	"os"
	"os/exec"
	"runtime"
	"strings"
)

// MenuUtils 提供通用的菜单工具函数
type MenuUtils struct{}

// NewMenuUtils 创建菜单工具实例
func NewMenuUtils() *MenuUtils {
	return &MenuUtils{}
}

// Pause 暂停等待用户按键
func (u *MenuUtils) Pause() {
	fmt.Print("\n按回车键继续...")
	fmt.Scanln()
}

// GetUserInput 获取用户输入并验证范围
func (u *MenuUtils) GetUserInput(min, max int) int {
	input := bufio.NewReader(os.Stdin)
	for {
		inputStr, err := input.ReadString('\n')
		if err != nil {
			// 如果读取失败（如EOF），返回退出选项
			fmt.Printf("❌ 输入读取错误: %v\n", err)
			return max // 返回最大值，通常是退出选项
		}

		inputStr = strings.TrimSpace(inputStr)

		if inputStr == "" {
			fmt.Println("❌ 输入不能为空，请重新选择")
			continue
		}

		var choice int
		_, err = fmt.Sscanf(inputStr, "%d", &choice)
		if err != nil || choice < min || choice > max {
			fmt.Printf("❌ 输入错误，请输入 %d-%d 之间的数字\n", min, max)
			continue
		}

		return choice
	}
}

// GetUserInputWithPrompt 带提示的用户输入
func (u *MenuUtils) GetUserInputWithPrompt(prompt string, min, max int) int {
	fmt.Print(prompt)
	return u.GetUserInput(min, max)
}

// ConfirmOperation 确认操作
func (u *MenuUtils) ConfirmOperation(message string) bool {
	fmt.Printf("%s (y/n): ", message)
	var confirm string
	_, err := fmt.Scanln(&confirm)
	if err != nil || (confirm != "y" && confirm != "Y") {
		return false
	}
	return true
}

// DisplayError 显示错误信息并暂停
func (u *MenuUtils) DisplayError(message string) {
	fmt.Printf("❌ %s\n", message)
	u.Pause()
}

// DisplaySuccess 显示成功信息并暂停
func (u *MenuUtils) DisplaySuccess(message string) {
	fmt.Printf("✅ %s\n", message)
	u.Pause()
}

// DisplayWarning 显示警告信息并暂停
func (u *MenuUtils) DisplayWarning(message string) {
	fmt.Printf("⚠️ %s\n", message)
	u.Pause()
}

// DisplayInfo 显示信息并暂停
func (u *MenuUtils) DisplayInfo(message string) {
	fmt.Printf("ℹ️ %s\n", message)
	u.Pause()
}

// ReadString 读取字符串输入
func (u *MenuUtils) ReadString(prompt string) string {
	fmt.Print(prompt)
	input := bufio.NewReader(os.Stdin)
	result, _ := input.ReadString('\n')
	return strings.TrimSpace(result)
}

// ReadStringWithDefault 读取字符串输入，支持默认值
func (u *MenuUtils) ReadStringWithDefault(prompt, defaultValue string) string {
	fmt.Printf("%s [默认: %s]: ", prompt, defaultValue)
	input := bufio.NewReader(os.Stdin)
	result, _ := input.ReadString('\n')
	result = strings.TrimSpace(result)

	if result == "" {
		return defaultValue
	}
	return result
}

// ValidateChoice 验证选择是否在范围内
func (u *MenuUtils) ValidateChoice(choice, min, max int) bool {
	return choice >= min && choice <= max
}

// PrintSeparator 打印分隔线
func (u *MenuUtils) PrintSeparator() {
	fmt.Println("============================================================")
}

// PrintSection 打印带标题的分隔区域
func (u *MenuUtils) PrintSection(title string) {
	fmt.Println(title)
	u.PrintSeparator()
}

// ClearScreen 清屏功能，跨平台支持
func (u *MenuUtils) ClearScreen() {
	var cmd *exec.Cmd

	if runtime.GOOS == "windows" {
		cmd = exec.Command("cmd", "/c", "cls")
	} else {
		cmd = exec.Command("clear")
	}

	cmd.Stdout = os.Stdout
	cmd.Run()
}
