package menu

import (
	"context"
	"fmt"
	"time"

	"index-go/config"
	"index-go/core"
	"index-go/daemon"
	"index-go/utils"
)

// MainMenu 主菜单结构
type MainMenu struct {
	cfg       *config.Config
	menuUtils *MenuUtils
}

// NewMainMenu 创建主菜单实例
func NewMainMenu(cfg *config.Config) *MainMenu {
	return &MainMenu{
		cfg:       cfg,
		menuUtils: NewMenuUtils(),
	}
}

// Show 显示主菜单并处理用户交互
func (m *MainMenu) Show() {
	for {
		// 清屏并显示主菜单界面
		m.menuUtils.ClearScreen()
		fmt.Println("🚀 腾讯云 COS 索引生成工具 v1.0 - 管理面板")
		fmt.Println("============================================================")

		fmt.Println("系统状态 :", checkCosConnection(m.cfg))
		fmt.Println("Web 服务 :", utils.FormatWebAddress(m.cfg.WebServer.Address))
		prefix := m.cfg.Prefix
		if prefix == "" {
			prefix = "根目录"
		}
		fmt.Println("索引目录 :", prefix)
		fmt.Println("============================================================")

		m.showMenuOptions()
		choice := m.menuUtils.GetUserInput(1, 5)

		switch choice {
		case 1:
			m.executeIndexGeneration()
		case 2:
			m.manageDaemon()
		case 3:
			m.manageSystemService()
		case 4:
			m.manageConfig()
		case 5:
			fmt.Println("👋 退出管理面板")
			return
		}
	}
}

// showMenuOptions 显示菜单选项
func (m *MainMenu) showMenuOptions() {
	fmt.Println("请选择操作:")
	fmt.Println("  1. 刷新索引")
	fmt.Println("  2. 后台服务 (Daemon)")
	fmt.Println("  3. 系统守护 (Server)")
	fmt.Println("  4. 配置管理")
	fmt.Println("  5. 退出管理")
	fmt.Print("请输入选项 [1-5]: ")
}

// executeIndexGeneration 执行索引生成
func (m *MainMenu) executeIndexGeneration() {
	fmt.Println("🔄 正在刷新索引数据...")

	engine, err := core.NewIndexEngine(m.cfg)
	if err != nil {
		m.menuUtils.DisplayError(fmt.Sprintf("创建索引引擎失败: %v", err))
		return
	}

	ctx, cancel := context.WithTimeout(context.Background(), 2*time.Minute)
	defer cancel()
	if err := engine.Refresh(ctx); err != nil {
		m.menuUtils.DisplayError(fmt.Sprintf("刷新索引失败: %v", err))
		return
	}

	fmt.Println("✅ 索引刷新完成！")
	fmt.Printf("🌐 Web 服务地址: %s\n", utils.FormatWebAddress(m.cfg.WebServer.Address))
	m.menuUtils.Pause()
}

// startWebService 启动 Web 服务（等同 ./cos-index --run）
// manageConfig 管理配置
func (m *MainMenu) manageConfig() {
	configMenu := NewConfigMenu(m.cfg, func() {
		// 返回时不需要做任何特殊操作，主循环会自动重新显示
	})
	configMenu.Show()
}

// manageSystemService 系统服务管理
func (m *MainMenu) manageSystemService() {
	m.menuUtils.ClearScreen()
	serviceMenu := NewServiceMenu(m.cfg)
	if err := serviceMenu.Show(); err != nil {
		m.menuUtils.DisplayError(fmt.Sprintf("服务管理出错: %v", err))
	}
}

func (m *MainMenu) manageDaemon() {
	for {
		m.menuUtils.ClearScreen()
		fmt.Println("后台服务管理")
		fmt.Println("============================================================")
		fmt.Printf("当前状态: %s\n", daemon.Status())
		fmt.Println("============================================================")
		fmt.Println("请选择操作:")
		fmt.Println("  1. 启动后台服务")
		fmt.Println("  2. 停止后台服务")
		fmt.Println("  3. 重启后台服务")
		fmt.Println("  4. 返回主菜单")
		fmt.Print("请输入选项 [1-4]: ")

		choice := m.menuUtils.GetUserInput(1, 4)

		switch choice {
		case 1:
			if err := daemon.Start(m.cfg); err != nil {
				m.menuUtils.DisplayError(err.Error())
			} else {
				m.menuUtils.DisplaySuccess("后台服务已启动")
			}
		case 2:
			if err := daemon.Stop(); err != nil {
				m.menuUtils.DisplayError(err.Error())
			} else {
				m.menuUtils.DisplaySuccess("后台服务已停止")
			}
		case 3:
			if err := daemon.Restart(m.cfg); err != nil {
				m.menuUtils.DisplayError(err.Error())
			} else {
				m.menuUtils.DisplaySuccess("后台服务已重启")
			}
		case 4:
			return
		}
	}
}
