package menu

import (
	"bufio"
	"fmt"
	"os"
	"strings"

	"index-go/config"
	"index-go/utils"
)

// ConfigMenu 配置管理菜单
type ConfigMenu struct {
	cfg       *config.Config
	menuUtils *MenuUtils
	onReturn  func()
}

// NewConfigMenu 创建配置管理菜单实例
func NewConfigMenu(cfg *config.Config, onReturn func()) *ConfigMenu {
	return &ConfigMenu{
		cfg:       cfg,
		menuUtils: NewMenuUtils(),
		onReturn:  onReturn,
	}
}

// Show 显示配置管理菜单
func (c *ConfigMenu) Show() {
	// 清屏
	c.menuUtils.ClearScreen()
	fmt.Println("🔧 配置管理")
	fmt.Println("============================================================")

	// 显示当前配置摘要
	ShowConfigSummary(c.cfg)
	fmt.Println("============================================================")

	// 配置管理子菜单循环
	for {
		// 每次循环开始时清屏
		c.menuUtils.ClearScreen()
		fmt.Println("配置管理")
		fmt.Println("============================================================")

		// 显示当前配置摘要
		ShowConfigSummary(c.cfg)
		fmt.Println("============================================================")

		c.showMenuOptions()
		choice := c.menuUtils.GetUserInput(1, 4)

		switch choice {
		case 1:
			c.modifyIndexConfig()
		case 2:
			c.modifyDomainConfig()
		case 3:
			c.showFullConfig()
		case 4:
			fmt.Println("🔙 返回主菜单")
			if c.onReturn != nil {
				c.onReturn()
			}
			return
		}
	}
}

// showMenuOptions 显示菜单选项
func (c *ConfigMenu) showMenuOptions() {
	fmt.Println("请选择操作:")
	fmt.Println("  1. 修改文件前缀")
	fmt.Println("  2. 修改访问域名")
	fmt.Println("  3. 查看完整配置")
	fmt.Println("  4. 返回主菜单")
	fmt.Print("请输入选项 [1-4]: ")
}

// modifyIndexConfig 修改索引配置
func (c *ConfigMenu) modifyIndexConfig() {
	c.modifyPrefix()
}

// modifyPrefix 修改文件前缀
func (c *ConfigMenu) modifyPrefix() {
	fmt.Println("\n📂 修改文件前缀")
	fmt.Println("============================================================")
	fmt.Printf("当前文件前缀: %s\n", getPrefixStatus(c.cfg.Prefix))
	fmt.Println("💡 文件前缀用于过滤特定目录下的文件，留空表示扫描根目录")
	fmt.Println("============================================================")

	fmt.Println("请输入新的文件前缀 (留空则扫描根目录):")
	fmt.Println("格式示例: documents/, images/2023/")
	fmt.Print("新前缀: ")

	input := bufio.NewReader(os.Stdin)
	newPrefix, _ := input.ReadString('\n')
	newPrefix = strings.TrimSpace(newPrefix)

	// 确认修改
	if newPrefix != "" {
		fmt.Printf("⚠️ 确认将文件前缀修改为: %s ？ (y/n): ", newPrefix)
	} else {
		fmt.Println("⚠️ 确认要清除文件前缀，扫描根目录？ (y/n): ")
	}

	var confirm string
	_, err := fmt.Scanln(&confirm)
	if err != nil || (confirm != "y" && confirm != "Y") {
		fmt.Println("❌ 操作已取消")
		c.menuUtils.Pause()
		return
	}

	// 保存旧值用于回滚
	oldPrefix := c.cfg.Prefix

	// 更新配置
	c.cfg.Prefix = newPrefix

	// 保存配置
	if err := config.SaveConfig(c.cfg); err != nil {
		fmt.Printf("❌ 保存配置失败: %v\n", err)
		// 恢复旧值
		c.cfg.Prefix = oldPrefix
		c.menuUtils.Pause()
		return
	}

	// 显示结果
	if newPrefix != "" {
		fmt.Printf("✅ 文件前缀已更新为: %s\n", newPrefix)
		fmt.Println("💡 将只扫描此前缀下的文件生成索引")
	} else {
		fmt.Println("✅ 文件前缀已清除")
		fmt.Println("💡 将扫描存储桶根目录下的所有文件")
	}

	c.menuUtils.Pause()
}

// modifyDomainConfig 修改域名配置
func (c *ConfigMenu) modifyDomainConfig() {
	fmt.Println("🌐 修改访问域名配置")
	fmt.Println("============================================================")

	// 显示当前域名配置
	if c.cfg.CustomDomain != "" {
		fmt.Printf("🌐 当前自定义域名: %s\n", c.cfg.CustomDomain)
		fmt.Println("💡 使用自定义域名可以提供更好的访问体验")
	} else {
		fmt.Println("🌐 当前未配置自定义域名")
		fmt.Println("💡 将使用腾讯云COS默认域名访问文件")
	}
	fmt.Println("============================================================")

	// 获取用户输入
	fmt.Println("请输入新的自定义域名 (留空则清除自定义域名):")
	fmt.Println("格式示例: https://cdn.example.com")
	fmt.Print("新域名: ")

	input := bufio.NewReader(os.Stdin)
	newDomain, _ := input.ReadString('\n')
	newDomain = strings.TrimSpace(newDomain)

	// 验证输入
	if newDomain != "" {
		// 简单的域名格式验证
		if !strings.HasPrefix(newDomain, "http://") && !strings.HasPrefix(newDomain, "https://") {
			fmt.Println("❌ 域名必须以 http:// 或 https:// 开头")
			c.menuUtils.Pause()
			return
		}

		// 确认修改
		fmt.Printf("⚠️ 确认将自定义域名修改为: %s ？ (y/n): ", newDomain)
		var confirm string
		_, err := fmt.Scanln(&confirm)
		if err != nil || (confirm != "y" && confirm != "Y") {
			fmt.Println("❌ 操作已取消")
			c.menuUtils.Pause()
			return
		}
	} else {
		// 清除域名的确认
		fmt.Println("⚠️ 确认要清除自定义域名，使用COS默认域名？ (y/n): ")
		var confirm string
		_, err := fmt.Scanln(&confirm)
		if err != nil || (confirm != "y" && confirm != "Y") {
			fmt.Println("❌ 操作已取消")
			c.menuUtils.Pause()
			return
		}
	}

	// 保存旧值用于回滚
	oldDomain := c.cfg.CustomDomain

	// 更新配置
	c.cfg.CustomDomain = newDomain

	// 保存配置
	if err := config.SaveConfig(c.cfg); err != nil {
		fmt.Printf("❌ 保存配置失败: %v\n", err)
		// 恢复旧值
		c.cfg.CustomDomain = oldDomain
		c.menuUtils.Pause()
		return
	}

	// 显示结果
	if newDomain != "" {
		fmt.Printf("✅ 自定义域名已更新为: %s\n", newDomain)
		fmt.Println("💡 新域名将在下次生成索引时生效")
	} else {
		fmt.Println("✅ 自定义域名已清除")
		fmt.Println("💡 将使用腾讯云COS默认域名访问文件")
	}

	c.menuUtils.Pause()
}

// showFullConfig 显示完整配置
func (c *ConfigMenu) showFullConfig() {
	fmt.Println("完整配置信息")
	fmt.Println("============================================================")

	fmt.Printf("🔑 腾讯云访问凭证:\n")
	fmt.Printf("   SecretId: %s\n", maskSecret(c.cfg.SecretId))
	fmt.Printf("   SecretKey: %s\n", maskSecret(c.cfg.SecretKey))

	fmt.Printf("\n🪣 COS存储桶配置:\n")
	fmt.Printf("   存储桶名称: %s\n", c.cfg.Bucket)
	fmt.Printf("   所在区域: %s\n", c.cfg.Region)

	fmt.Printf("\n📁 文件过滤配置:\n")
	fmt.Printf("   文件前缀: %s\n", getPrefixStatus(c.cfg.Prefix))

	fmt.Printf("\n🌐 自定义域名配置:\n")
	fmt.Printf("   自定义域名: %s\n", getDomainStatus(c.cfg.CustomDomain))
	fmt.Printf("\n🌐 Web 服务配置:\n")
	fmt.Printf("   监听地址: %s\n", c.cfg.WebServer.Address)
	fmt.Printf("   浏览地址: %s\n", utils.FormatWebAddress(c.cfg.WebServer.Address))

	fmt.Println("\n============================================================")
	fmt.Printf("📁 配置文件位置: %s/%s\n", config.ConfigDirName, config.ConfigFileName)

	c.menuUtils.Pause()
}

// getPrefixStatus 获取前缀状态
func getPrefixStatus(prefix string) string {
	if prefix == "" {
		return "未设置 (扫描根目录)"
	}
	return prefix
}
