package main

import (
	"fmt"
	"os"

	"index-go/config"
	"index-go/daemon"
	"index-go/menu"
	"index-go/service"
	"index-go/utils"
	"index-go/webserver"
)

func main() {
	// 检查命令行参数
	if len(os.Args) < 2 {
		// 默认打开管理面板
		runManageMode()
		return
	}

	switch os.Args[1] {
	case "--help", "-h":
		showHelp()
		return
	case "--version", "-v":
		fmt.Printf("腾讯云 COS 索引生成工具 v1.0.0\n")
		return
	case "--run":
		runDirectMode()
		return
	case "--daemon":
		runDaemonCommand(os.Args[2:])
		return
	case "--daemon-run":
		runDaemonWorker()
		return
	case "--server":
		runServerCommand(os.Args[2:])
		return
	case "--manage":
		runManageMode()
		return
	default:
		fmt.Printf("未知参数: %s\n", os.Args[1])
		showHelp()
		return
	}
}

// showHelp 显示帮助信息
func showHelp() {
	fmt.Println("腾讯云 COS 索引生成工具 v1.0.0")
	fmt.Println()
	fmt.Println("用法:")
	fmt.Println("  ./cos-index                 打开管理面板（默认）")
	fmt.Println("  ./cos-index --daemon CMD   管理后台 Web 服务 (start|stop|restart|status)")
	fmt.Println("  ./cos-index --server CMD   管理系统守护 (start|stop|restart|status|install|uninstall)")
	fmt.Println("  ./cos-index --run          兼容模式，直接启动 Web 服务")
	fmt.Println("  ./cos-index --manage       打开管理面板")
	fmt.Println("  ./cos-index --help         显示帮助信息")
	fmt.Println("  ./cos-index --version      显示版本信息")
	fmt.Println()
	fmt.Println("功能说明:")
	fmt.Println("  该工具扫描腾讯云 COS 存储桶中的文件，并通过内置 Web 服务展示目录索引")
	fmt.Println("  不再上传静态 HTML 文件，直接访问 http://localhost:8888（默认）浏览")
	fmt.Println()
	fmt.Println("管理面板:")
	fmt.Println("  • 直接运行 ./cos-index 或使用 --manage : 打开可视化管理界面，可查看状态、管理配置等")
	fmt.Println()
	fmt.Println("配置文件:")
	fmt.Println("  配置文件位置: app_config/config.toml")
	fmt.Println("  首次运行会自动创建默认配置文件")
	fmt.Println("  请根据实际情况修改配置文件中的参数")
	fmt.Println()
	fmt.Println("必要配置项:")
	fmt.Println("  • SecretId  - 腾�讯云访问密钥 ID")
	fmt.Println("  • SecretKey - 腾讯云访问密钥 Key")
	fmt.Println("  • Bucket    - COS 存储桶名称")
	fmt.Println("  • Region     - COS 所在区域")
	fmt.Println()
	fmt.Println("定时任务配置:")
	fmt.Println("  • webserver.address - Web 服务监听地址，默认 127.0.0.1:8888")
}

// runDirectMode 启动内置 Web 服务
func runDirectMode() {
	// 加载配置
	cfg, err := config.LoadConfig()
	if err != nil {
		fmt.Printf("❌ 配置加载失败: %v\n", err)
		if _, err = config.CreateDefaultConfig(); err != nil {
			fmt.Printf("❌ 创建默认配置失败: %v\n", err)
		} else {
			fmt.Println("✅ 默认配置文件已创建: app_config/config.toml")
			fmt.Println("请修改配置文件后重新运行")
		}
		return
	}

	fmt.Println("✅ 配置加载成功")

	if err := webserver.Run(cfg); err != nil {
		fmt.Printf("❌ Web 服务异常退出: %v\n", err)
	}
}

func runDaemonWorker() {
	cfg, err := config.LoadConfig()
	if err != nil {
		fmt.Printf("❌ 配置加载失败: %v\n", err)
		return
	}

	if err := webserver.Run(cfg); err != nil {
		fmt.Printf("❌ Web 服务异常退出: %v\n", err)
	}
}

func runDaemonCommand(args []string) {
	if len(args) == 0 {
		fmt.Println("用法: ./cos-index --daemon [start|stop|restart|status]")
		return
	}

	switch args[0] {
	case "start":
		cfg, err := config.LoadConfig()
		if err != nil {
			fmt.Printf("❌ 配置加载失败: %v\n", err)
			return
		}
		if err := daemon.Start(cfg); err != nil {
			fmt.Printf("❌ 启动失败: %v\n", err)
		} else {
			fmt.Println("✅ 后台服务已启动")
		}
	case "stop":
		if err := daemon.Stop(); err != nil {
			fmt.Printf("❌ 停止失败: %v\n", err)
		} else {
			fmt.Println("✅ 后台服务已停止")
		}
	case "restart":
		cfg, err := config.LoadConfig()
		if err != nil {
			fmt.Printf("❌ 配置加载失败: %v\n", err)
			return
		}
		if err := daemon.Restart(cfg); err != nil {
			fmt.Printf("❌ 重启失败: %v\n", err)
		} else {
			fmt.Println("✅ 后台服务已重启")
		}
	case "status":
		fmt.Println(daemon.Status())
	default:
		fmt.Println("未知命令，使用 start|stop|restart|status")
	}
}

func runServerCommand(args []string) {
	if len(args) == 0 {
		fmt.Println("用法: ./cos-index --server [start|stop|restart|status|install|uninstall]")
		return
	}

	cfg, _ := config.LoadConfig()
	mgr, err := service.CreateServiceManager()
	if err != nil {
		fmt.Printf("❌ 当前平台不支持系统守护: %v\n", err)
		return
	}

	switch args[0] {
	case "start":
		if err := daemon.EnsureStopped(); err != nil {
			fmt.Printf("⚠️ 停止后台服务失败: %v\n", err)
		}
		if !mgr.IsServiceInstalled() {
			if err := mgr.InstallService(); err != nil {
				fmt.Printf("❌ 安装失败: %v\n", err)
				return
			}
		}
		if err := mgr.StartService(); err != nil {
			fmt.Printf("❌ 启动失败: %v\n", err)
		} else {
			fmt.Println("✅ 系统守护已启动")
		}
	case "stop":
		if err := mgr.StopService(); err != nil {
			fmt.Printf("❌ 停止失败: %v\n", err)
		} else {
			fmt.Println("✅ 系统守护已停止")
		}
	case "restart":
		if err := mgr.RestartService(); err != nil {
			fmt.Printf("❌ 重启失败: %v\n", err)
		} else {
			fmt.Println("✅ 系统守护已重启")
		}
	case "status":
		fmt.Println(service.FormatServiceStatus(mgr.GetServiceStatus()))
	case "install":
		if err := daemon.EnsureStopped(); err != nil {
			fmt.Printf("⚠️ 停止后台服务失败: %v\n", err)
		}
		if err := mgr.InstallService(); err != nil {
			fmt.Printf("❌ 安装失败: %v\n", err)
		} else {
			fmt.Println("✅ 系统守护已安装")
		}
	case "uninstall":
		_ = mgr.StopService()
		if err := mgr.UninstallService(); err != nil {
			fmt.Printf("❌ 卸载失败: %v\n", err)
			return
		}
		fmt.Println("✅ 系统守护已卸载，准备恢复后台模式")
		if cfg != nil {
			if err := daemon.Start(cfg); err != nil {
				fmt.Printf("⚠️ 启动后台服务失败: %v\n", err)
			} else {
				fmt.Println("✅ 已切换回后台服务模式")
			}
		}
	default:
		fmt.Println("未知命令，使用 start|stop|restart|status|install|uninstall")
	}
}

// runManageMode 管理面板模式
func runManageMode() {
	// 初始化日志系统
	if err := utils.InitDefaultLogger(); err != nil {
		fmt.Printf("❌ 日志初始化失败: %v\n", err)
	} else {
		utils.PanelLog("启动管理面板")
	}

	// 加载配置
	cfg, err := config.LoadConfig()
	if err != nil {
		fmt.Printf("❌ 配置加载失败: %v\n", err)
		return
	}

	// 创建并显示主菜单
	mainMenu := menu.NewMainMenu(cfg)
	mainMenu.Show()
}
