package config

import (
	"fmt"
	"strings"
)

// validateConfig 验证配置的有效性
func ValidateConfig(config *Config) error {
	if config.SecretId == "" {
		return fmt.Errorf("配置错误: SecretId 不能为空")
	}
	if config.SecretKey == "" {
		return fmt.Errorf("配置错误: SecretKey 不能为空")
	}
	if config.Bucket == "" {
		return fmt.Errorf("配置错误: Bucket 不能为空")
	}
	if config.Region == "" {
		return fmt.Errorf("配置错误: Region 不能为空")
	}

	// 验证字段长度和格式
	if len(config.SecretId) < 10 {
		return fmt.Errorf("配置错误: SecretId 长度无效")
	}
	if len(config.SecretKey) < 10 {
		return fmt.Errorf("配置错误: SecretKey 长度无效")
	}
	if len(config.Bucket) < 3 || len(config.Bucket) > 63 {
		return fmt.Errorf("配置错误: Bucket 名称长度应在3-63字符之间")
	}

	// 验证 Region 格式
	if !IsValidRegion(config.Region) {
		return fmt.Errorf("配置错误: Region 格式无效，应为 'ap-城市' 格式")
	}

	// 验证自定义域名格式
	if config.CustomDomain != "" {
		if !IsValidDomain(config.CustomDomain) {
			return fmt.Errorf("配置错误: CustomDomain 格式无效")
		}
	}

	if strings.TrimSpace(config.WebServer.Address) == "" {
		config.WebServer.Address = DefaultWebAddress
	}
	if !strings.Contains(config.WebServer.Address, ":") {
		return fmt.Errorf("配置错误: webserver.address 必须包含端口, 例如 127.0.0.1:8888")
	}

	return nil
}

// IsValidRegion 验证区域格式
func IsValidRegion(region string) bool {
	if len(region) < 3 {
		return false
	}
	if !strings.HasPrefix(region, "ap-") && !strings.HasPrefix(region, "na-") && !strings.HasPrefix(region, "eu-") {
		return false
	}
	return true
}

// IsValidDomain 验证域名格式
func IsValidDomain(domain string) bool {
	if len(domain) < 4 || len(domain) > 253 {
		return false
	}

	// 移除协议前缀
	domain = strings.TrimPrefix(domain, "http://")
	domain = strings.TrimPrefix(domain, "https://")
	domain = strings.TrimSuffix(domain, "/")

	// 基本域名格式检查
	if !strings.Contains(domain, ".") {
		return false
	}

	// 检查是否包含非法字符
	for _, char := range domain {
		if !((char >= 'a' && char <= 'z') || (char >= '0' && char <= '9') || char == '.' || char == '-') {
			return false
		}
	}

	return true
}
