package config

import (
	"fmt"
	"log"
	"os"
	"path/filepath"

	"github.com/BurntSushi/toml"
)

// LoadConfig 加载或创建配置文件。
func LoadConfig() (*Config, error) {
	configPath := filepath.Join(ConfigDirName, ConfigFileName)
	if _, err := os.Stat(configPath); os.IsNotExist(err) {
		log.Printf("未找到配置文件，正在创建默认配置...")
		if err := os.MkdirAll(ConfigDirName, 0755); err != nil {
			return nil, fmt.Errorf("创建配置目录失败: %w", err)
		}

		// 创建带注释的 TOML 配置文件
		tomlContent := `# 腾讯云对象存储（COS）索引生成工具配置文件
# 请根据您的实际配置修改以下信息

# ===========================================
# 腾讯云访问凭证配置
# ===========================================
# 在腾讯云控制台 > 访问管理 > API密钥管理 中获取
SecretId  = ""  # 您的腾讯云 Secret ID（必填）
SecretKey = ""  # 您的腾讯云 Secret Key（必填）

# ===========================================
# COS 存储桶配置
# ===========================================
Bucket = ""           # 您的 COS 存储桶名称（必填）
Region = "ap-beijing" # COS 所在区域，如：ap-beijing, ap-shanghai, ap-guangzhou

# ===========================================
# 文件过滤和访问配置
# ===========================================
Prefix       = ""     # 文件路径前缀，留空表示扫描整个存储桶
CustomDomain = ""     # 自定义域名，用于文件访问 URL（可选）

# ===========================================
# Web 服务配置
# ===========================================
[webserver]
address = "127.0.0.1:8888"  # Web 服务监听地址，默认为本地 8888 端口
`

		os.WriteFile(configPath, []byte(tomlContent), 0600)
		log.Printf("默认配置文件已创建，请修改后重新运行。")
		return nil, fmt.Errorf("请更新配置文件")
	}

	data, err := os.ReadFile(configPath)
	if err != nil {
		return nil, err
	}

	var config Config
	if err := toml.Unmarshal(data, &config); err != nil {
		return nil, fmt.Errorf("解析配置文件失败: %w", err)
	}

	// 验证必填字段
	if err := ValidateConfig(&config); err != nil {
		return nil, err
	}

	return &config, nil
}

// SaveConfig 保存配置到文件
func SaveConfig(config *Config) error {
	data, err := toml.Marshal(config)
	if err != nil {
		return fmt.Errorf("序列化配置失败: %w", err)
	}

	configPath := filepath.Join(ConfigDirName, ConfigFileName)
	if err := os.WriteFile(configPath, data, 0600); err != nil {
		return fmt.Errorf("写入配置文件失败: %w", err)
	}

	return nil
}

// CreateDefaultConfig 创建默认配置并保存（带注释）
func CreateDefaultConfig() (*Config, error) {
	if err := os.MkdirAll(ConfigDirName, 0755); err != nil {
		return nil, fmt.Errorf("创建配置目录失败: %w", err)
	}

	// 使用带注释的 TOML 模板创建配置文件
	configPath := GetConfigPath()

	// 创建带注释的 TOML 配置文件
	tomlContent := `# 腾讯云对象存储（COS）索引生成工具配置文件
# 请根据您的实际配置修改以下信息

# ===========================================
# 腾讯云访问凭证配置
# ===========================================
# 在腾讯云控制台 > 访问管理 > API密钥管理 中获取
SecretId  = ""  # 您的腾讯云 Secret ID（必填）
SecretKey = ""  # 您的腾讯云 Secret Key（必填）

# ===========================================
# COS 存储桶配置
# ===========================================
Bucket = ""           # 您的 COS 存储桶名称（必填）
Region = "ap-beijing" # COS 所在区域，如：ap-beijing, ap-shanghai, ap-guangzhou

# ===========================================
# 文件过滤和访问配置
# ===========================================
Prefix       = ""     # 文件路径前缀，留空表示扫描整个存储桶
CustomDomain = ""     # 自定义域名，用于文件访问 URL（可选）

# ===========================================
# Web 服务配置
# ===========================================
[webserver]
address = "127.0.0.1:8888"  # Web 服务监听地址，默认为本地 8888 端口
`

	if err := os.WriteFile(configPath, []byte(tomlContent), 0600); err != nil {
		return nil, fmt.Errorf("写入默认配置文件失败: %w", err)
	}

	// 解析创建的配置文件以返回Config对象
	data, err := os.ReadFile(configPath)
	if err != nil {
		return nil, fmt.Errorf("读取配置文件失败: %w", err)
	}

	var config Config
	if err := toml.Unmarshal(data, &config); err != nil {
		return nil, fmt.Errorf("解析配置文件失败: %w", err)
	}

	return &config, nil
}

// GetConfigPath 获取配置文件路径
func GetConfigPath() string {
	return filepath.Join(ConfigDirName, ConfigFileName)
}
